"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AgentWSClient = void 0;
const ws_1 = __importDefault(require("ws"));
const os = __importStar(require("os"));
const crypto = __importStar(require("crypto"));
const config_1 = require("../config");
const logger_1 = require("../logger");
const events_1 = require("events");
const os_1 = require("os");
const AGENT_VERSION = '1.4.9';
function getIPAddress() {
    const nets = (0, os_1.networkInterfaces)();
    for (const name of Object.keys(nets)) {
        for (const net of nets[name] || []) {
            if (net.family === 'IPv4' && !net.internal) {
                return net.address;
            }
        }
    }
    return '127.0.0.1';
}
function getMACAddress() {
    const nets = (0, os_1.networkInterfaces)();
    for (const name of Object.keys(nets)) {
        for (const net of nets[name] || []) {
            if (net.family === 'IPv4' && !net.internal && net.mac && net.mac !== '00:00:00:00:00:00') {
                return net.mac;
            }
        }
    }
    return '00:00:00:00:00:00';
}
function getAgentId() {
    // Use hostname as the agent ID for backward compatibility and readability
    // Falls back to hash-based ID only if hostname is empty
    const hostname = os.hostname();
    if (hostname)
        return hostname;
    const mac = getMACAddress();
    return crypto.createHash('sha256').update(`${hostname}-${mac}`).digest('hex').substring(0, 32);
}
class AgentWSClient extends events_1.EventEmitter {
    constructor() {
        super();
        this.ws = null;
        this.reconnectTimer = null;
        this.heartbeatTimer = null;
        this.pingTimer = null;
        this.reconnectDelay = 1000;
        this.maxReconnectDelay = 30000;
        this.connected = false;
        this.agentId = null;
        this.serverConfig = {};
    }
    connect() {
        if (this.ws) {
            this.ws.removeAllListeners('close');
            try {
                this.ws.close();
            }
            catch { }
        }
        const url = config_1.config.server.wsUrl;
        logger_1.logger.info(`Connecting to server: ${url}`);
        this.ws = new ws_1.default(url, {
            headers: {
                'X-Agent-Token': config_1.config.server.token,
                'X-Agent-Hostname': config_1.config.agent.hostname,
                'X-Agent-Platform': config_1.config.agent.platform,
                'X-Agent-Version': AGENT_VERSION,
            },
            perMessageDeflate: true,
        });
        this.ws.on('open', () => {
            logger_1.logger.info('WebSocket connected');
            this.connected = true;
            this.reconnectDelay = 1000;
            this.register();
            this.startHeartbeat();
            this.emit('connected');
        });
        this.ws.on('message', (data) => {
            try {
                const message = JSON.parse(data.toString());
                this.handleMessage(message);
            }
            catch (e) {
                logger_1.logger.error('Failed to parse message:', e);
            }
        });
        this.ws.on('close', (code, reason) => {
            const reasonText = reason?.toString() || '';
            logger_1.logger.warn(`WebSocket closed: ${code} - ${reasonText}`);
            this.connected = false;
            this.stopHeartbeat();
            const duplicateConflict = code === 4001
                || code === 4002
                || /duplicate|older|newer|replaced/i.test(reasonText);
            if (duplicateConflict) {
                if (this.reconnectTimer) {
                    clearTimeout(this.reconnectTimer);
                    this.reconnectTimer = null;
                }
                logger_1.logger.error('Duplicate/superseded agent instance detected; reconnect suppressed');
                this.emit('duplicate-instance', { code, reason: reasonText });
                return;
            }
            this.scheduleReconnect();
            this.emit('disconnected');
        });
        this.ws.on('error', (error) => {
            logger_1.logger.error('WebSocket error:', error);
        });
        this.ws.on('pong', () => {
            // Connection alive
        });
    }
    register() {
        const userInfo = os.userInfo();
        this.send({
            type: 'agent:register',
            payload: {
                id: getAgentId(),
                hostname: config_1.config.agent.hostname,
                platform: config_1.config.agent.platform,
                arch: config_1.config.agent.arch,
                osVersion: `${os.type()} ${os.release()}`,
                agentVersion: AGENT_VERSION,
                ipAddress: getIPAddress(),
                macAddress: getMACAddress(),
                username: userInfo.username,
                domain: process.env.USERDOMAIN || process.env.HOSTNAME || null,
                tags: config_1.config.agent.tags,
                groups: config_1.config.agent.groups,
                capabilities: {
                    systemMonitoring: true,
                    processMonitoring: true,
                    networkMonitoring: true,
                    productivityTracking: true,
                    granularProductivityTracking: true,
                    browserMonitoring: true,
                    mtrProbes: true,
                    pingProbes: true,
                    trafficFiltering: true,
                    remoteTermination: true,
                    selfUpgrade: true,
                    wireGuard: true,
                    screenshotCapture: true,
                },
                systemInfo: {
                    totalMemory: os.totalmem(),
                    cpuCount: os.cpus().length,
                    cpuModel: os.cpus()[0]?.model || 'Unknown',
                    uptime: os.uptime(),
                    username: userInfo.username,
                    homeDir: os.homedir(),
                },
            },
        });
    }
    handleMessage(message) {
        logger_1.logger.debug(`Received: ${message.type}`);
        switch (message.type) {
            case 'agent:ack':
            case 'registration:ack':
                this.agentId = message.agentId || message.payload?.agentId || getAgentId();
                this.serverConfig = message.payload?.config || {};
                logger_1.logger.info(`Registered as agent: ${this.agentId}`);
                this.emit('registered', this.agentId, this.serverConfig);
                break;
            case 'config:update':
                this.serverConfig = { ...this.serverConfig, ...message.payload };
                logger_1.logger.info('Configuration updated from server');
                this.emit('config-update', this.serverConfig);
                break;
            case 'process:terminate':
                this.emit('process-terminate', message.payload);
                break;
            case 'monitoring:toggle':
                this.emit('monitoring-toggle', message.payload);
                break;
            case 'probe:start':
            case 'probe:config:update':
                this.emit('probe-start', message.payload);
                break;
            case 'probe:stop':
                this.emit('probe-stop', message.payload);
                break;
            case 'upgrade:command':
            case 'upgrade:force':
                this.emit('upgrade-command', message.payload);
                break;
            case 'upgrade:available':
                this.emit('upgrade-available', message.payload);
                break;
            case 'traffic:rule:update':
            case 'traffic:rules-update':
                this.emit('traffic-rules-update', message.payload);
                break;
            case 'url:filter:update':
            case 'traffic:filters-update':
                this.emit('traffic-filters-update', message.payload);
                break;
            case 'system:metrics:request':
            case 'request:system-info':
                this.emit('request-system-info', message.payload);
                break;
            case 'process:list:request':
            case 'request:processes':
                this.emit('request-processes', message.payload);
                break;
            case 'software:list:request':
            case 'request:software':
                this.emit('request-software', message.payload);
                break;
            case 'software:uninstall':
            case 'software:remove':
                this.emit('software-uninstall', message.payload);
                break;
            case 'extension:install':
            case 'browser:extension:install':
                this.emit('extension-install', message.payload);
                break;
            case 'security:scan:start':
            case 'security:scan':
                this.emit('security-scan', message.payload);
                break;
            case 'remote:session:start':
            case 'remote:session':
                this.emit('remote-session-start', message.payload);
                break;
            case 'remote:session:end':
                this.emit('remote-session-end', message.payload);
                break;
            case 'wireguard:config':
                this.emit('wireguard-config', message.payload);
                break;
            case 'speedtest:run':
            case 'speedtest:request':
                this.emit('speedtest-run', message.payload);
                break;
            default:
                logger_1.logger.debug(`Unhandled message type: ${message.type}`);
                this.emit('message', message);
        }
    }
    send(message) {
        if (!this.ws || !this.connected) {
            logger_1.logger.warn('Cannot send: not connected');
            return;
        }
        try {
            this.ws.send(JSON.stringify({
                ...message,
                timestamp: new Date().toISOString(),
            }));
        }
        catch (e) {
            logger_1.logger.error('Send error:', e);
        }
    }
    sendMetrics(type, data) {
        this.send({ type, payload: { agentId: this.agentId, ...data } });
    }
    startHeartbeat() {
        this.stopHeartbeat();
        // Application-level heartbeat
        this.heartbeatTimer = setInterval(() => {
            this.send({
                type: 'agent:heartbeat',
                payload: {
                    agentId: this.agentId || getAgentId(),
                    uptime: process.uptime(),
                    memoryUsage: process.memoryUsage(),
                    version: AGENT_VERSION,
                    status: 'healthy',
                    activeCollectors: ['system', 'process', 'productivity', 'network'],
                    pendingDataCount: 0,
                    errors: [],
                    timestamp: new Date().toISOString(),
                },
            });
        }, config_1.config.intervals.heartbeat);
        // WebSocket-level ping
        this.pingTimer = setInterval(() => {
            if (this.ws?.readyState === ws_1.default.OPEN) {
                this.ws.ping();
            }
        }, 15000);
    }
    stopHeartbeat() {
        if (this.heartbeatTimer) {
            clearInterval(this.heartbeatTimer);
            this.heartbeatTimer = null;
        }
        if (this.pingTimer) {
            clearInterval(this.pingTimer);
            this.pingTimer = null;
        }
    }
    scheduleReconnect() {
        if (this.reconnectTimer)
            return;
        // Don't schedule reconnect if we're already connected (prevents stale close handlers from looping)
        if (this.connected)
            return;
        logger_1.logger.info(`Reconnecting in ${this.reconnectDelay / 1000}s...`);
        this.reconnectTimer = setTimeout(() => {
            this.reconnectTimer = null;
            this.reconnectDelay = Math.min(this.reconnectDelay * 2, this.maxReconnectDelay);
            this.connect();
        }, this.reconnectDelay);
    }
    isConnected() {
        return this.connected;
    }
    getAgentId() {
        return this.agentId;
    }
    getServerConfig() {
        return this.serverConfig;
    }
    disconnect() {
        this.stopHeartbeat();
        if (this.reconnectTimer) {
            clearTimeout(this.reconnectTimer);
            this.reconnectTimer = null;
        }
        if (this.ws) {
            this.ws.close(1000, 'Agent shutting down');
            this.ws = null;
        }
        this.connected = false;
    }
    /**
     * Reconnect the WebSocket using the current config.server.wsUrl.
     * Call this after updating config.server.wsUrl at runtime (e.g. after
     * WireGuard tunnel is established) to route traffic through the tunnel.
     */
    reconnect() {
        logger_1.logger.info('WS reconnect requested — closing current connection');
        this.stopHeartbeat();
        if (this.reconnectTimer) {
            clearTimeout(this.reconnectTimer);
            this.reconnectTimer = null;
        }
        if (this.ws) {
            // Remove close listener first to prevent scheduleReconnect from firing
            this.ws.removeAllListeners('close');
            try {
                this.ws.close(1000, 'Reconnecting with updated URL');
            }
            catch { }
            this.ws = null;
        }
        this.connected = false;
        // Reset backoff so reconnect is immediate
        this.reconnectDelay = 1000;
        // connect() reads config.server.wsUrl fresh each time — picks up the new URL
        setTimeout(() => this.connect(), 500);
    }
}
exports.AgentWSClient = AgentWSClient;
//# sourceMappingURL=client.js.map