"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeService = void 0;
const fs = __importStar(require("fs"));
const path = __importStar(require("path"));
const crypto = __importStar(require("crypto"));
const http = __importStar(require("http"));
const https = __importStar(require("https"));
const child_process_1 = require("child_process");
const config_1 = require("../config");
const logger_1 = require("../logger");
const AGENT_VERSION = '1.4.9';
class UpgradeService {
    constructor(wsClient) {
        this.upgrading = false;
        this.pendingUpgrade = null;
        this.wsClient = wsClient;
    }
    // ---- Called when server notifies an upgrade is available ----
    handleUpgradeAvailable(info) {
        const version = info.targetVersion || info.version;
        logger_1.logger.info(`Upgrade available: v${AGENT_VERSION} -> v${version} (mandatory=${info.mandatory})`);
        if (this.upgrading) {
            logger_1.logger.warn('Upgrade already in progress, ignoring availability notice');
            return;
        }
        // If mandatory OR auto-update is enabled, proceed immediately
        if (info.mandatory) {
            logger_1.logger.info('Mandatory upgrade — proceeding automatically');
            this.pendingUpgrade = null;
            this.performUpgrade(info);
        }
        else if (config_1.config.upgrade.autoUpdate) {
            logger_1.logger.info('Auto-update enabled — accepting upgrade');
            this.pendingUpgrade = null;
            // Report acceptance then proceed
            this.reportUpgradeResponse('accepted', version);
            this.performUpgrade(info);
        }
        else {
            // Store as pending — server or admin can force later
            this.pendingUpgrade = info;
            logger_1.logger.info('Auto-update disabled — upgrade stored as pending. Use force to apply.');
            this.reportUpgradeResponse('deferred', version);
        }
    }
    // ---- Called when server sends a direct upgrade command (force or admin-triggered) ----
    async handleUpgradeCommand(command) {
        // Normalize to UpgradeInfo
        const info = {
            targetVersion: command.targetVersion || command.version,
            packageUrl: command.packageUrl || command.downloadUrl,
            checksum: command.checksum,
            mandatory: command.mandatory !== undefined ? command.mandatory : true, // commands default to mandatory
            releaseNotes: command.releaseNotes,
        };
        logger_1.logger.info(`Upgrade COMMAND received (forced): v${AGENT_VERSION} -> v${info.targetVersion}`);
        this.pendingUpgrade = null;
        await this.performUpgrade(info);
    }
    // ---- Accept a previously deferred upgrade ----
    acceptPendingUpgrade() {
        if (!this.pendingUpgrade) {
            logger_1.logger.warn('No pending upgrade to accept');
            return false;
        }
        const info = this.pendingUpgrade;
        this.pendingUpgrade = null;
        this.reportUpgradeResponse('accepted', info.targetVersion);
        this.performUpgrade(info);
        return true;
    }
    // ---- Reject a previously deferred upgrade ----
    rejectPendingUpgrade() {
        if (!this.pendingUpgrade) {
            logger_1.logger.warn('No pending upgrade to reject');
            return false;
        }
        const version = this.pendingUpgrade.targetVersion;
        this.pendingUpgrade = null;
        logger_1.logger.info(`Upgrade to v${version} rejected by agent`);
        this.reportUpgradeResponse('rejected', version);
        return true;
    }
    getPendingUpgrade() {
        return this.pendingUpgrade;
    }
    isUpgrading() {
        return this.upgrading;
    }
    // ---- Core upgrade logic ----
    async performUpgrade(info) {
        if (this.upgrading) {
            logger_1.logger.warn('Upgrade already in progress');
            return;
        }
        const version = info.targetVersion;
        this.upgrading = true;
        this.reportStatus('downloading', version, 0);
        try {
            // 1. Build full download URL
            const baseUrl = config_1.config.server.url.replace(/\/$/, '');
            const downloadUrl = info.packageUrl.startsWith('http')
                ? info.packageUrl
                : `${baseUrl}${info.packageUrl}`;
            logger_1.logger.info(`Downloading from: ${downloadUrl}`);
            // 2. Download the package
            const downloadPath = path.join(config_1.config.upgrade.installPath, 'updates');
            if (!fs.existsSync(downloadPath))
                fs.mkdirSync(downloadPath, { recursive: true });
            const ext = downloadUrl.endsWith('.zip') ? '.zip' : '.tar.gz';
            const packageFile = path.join(downloadPath, `appstats-agent-${version}${ext}`);
            await this.downloadPackage(downloadUrl, packageFile);
            this.reportStatus('downloading', version, 50);
            // 3. Verify checksum (skip if server didn't provide one)
            if (info.checksum) {
                this.reportStatus('verifying', version, 55);
                const fileChecksum = await this.calculateChecksum(packageFile);
                if (fileChecksum !== info.checksum) {
                    throw new Error(`Checksum mismatch: expected ${info.checksum}, got ${fileChecksum}`);
                }
                logger_1.logger.info('Checksum verified');
            }
            else {
                logger_1.logger.info('No checksum provided, skipping verification');
            }
            // 4. Extract
            this.reportStatus('extracting', version, 60);
            const extractDir = path.join(downloadPath, `v${version}`);
            if (fs.existsSync(extractDir)) {
                this.removeRecursive(extractDir);
            }
            await this.extractPackage(packageFile, extractDir);
            this.reportStatus('extracting', version, 75);
            // 5. Backup current version
            this.reportStatus('installing', version, 80);
            const backupDir = path.join(config_1.config.upgrade.installPath, 'backup', `v${AGENT_VERSION}`);
            await this.backupCurrent(backupDir);
            // 6. Replace files (best-effort while process is running)
            this.reportStatus('replacing', version, 85);
            try {
                await this.replaceFiles(extractDir, config_1.config.upgrade.installPath);
            }
            catch (replaceErr) {
                logger_1.logger.warn(`In-process file replace failed (will retry in restart script): ${replaceErr.message}`);
            }
            // 7. Clean up downloaded package (keep extractDir for restart script fallback)
            try {
                fs.unlinkSync(packageFile);
            }
            catch { }
            this.reportStatus('completed', version, 100);
            logger_1.logger.info(`Upgrade to v${version} completed successfully`);
            // 8. Restart — the restart script will also copy files from extractDir
            //    as a reliable fallback in case in-process replace failed
            this.reportStatus('restarting', version, 100);
            logger_1.logger.info('Restarting agent in 3 seconds...');
            setTimeout(() => {
                this.restartAgent(extractDir);
            }, 3000);
        }
        catch (error) {
            logger_1.logger.error(`Upgrade to v${version} failed:`, error);
            this.reportStatus('failed', version, 0, error.message);
            this.upgrading = false;
        }
    }
    // ---- Download using native http/https (no node-fetch dependency) ----
    downloadPackage(url, dest) {
        return new Promise((resolve, reject) => {
            const proto = url.startsWith('https') ? https : http;
            const fileStream = fs.createWriteStream(dest);
            let totalBytes = 0;
            const makeRequest = (requestUrl, redirectCount = 0) => {
                if (redirectCount > 5) {
                    reject(new Error('Too many redirects'));
                    return;
                }
                const urlObj = new URL(requestUrl);
                const options = {
                    hostname: urlObj.hostname,
                    port: urlObj.port || (requestUrl.startsWith('https') ? 443 : 80),
                    path: urlObj.pathname + urlObj.search,
                    method: 'GET',
                    headers: {
                        'X-Agent-Token': config_1.config.server.token,
                        'User-Agent': `AppStats-Agent/${AGENT_VERSION}`,
                    },
                };
                const reqProto = requestUrl.startsWith('https') ? https : http;
                reqProto.request(options, (response) => {
                    // Follow redirects
                    if (response.statusCode && response.statusCode >= 300 && response.statusCode < 400 && response.headers.location) {
                        makeRequest(response.headers.location, redirectCount + 1);
                        return;
                    }
                    if (response.statusCode !== 200) {
                        fileStream.destroy();
                        fs.unlink(dest, () => { });
                        reject(new Error(`Download failed: HTTP ${response.statusCode}`));
                        return;
                    }
                    response.on('data', (chunk) => {
                        totalBytes += chunk.length;
                    });
                    response.pipe(fileStream);
                    fileStream.on('finish', () => {
                        fileStream.close();
                        logger_1.logger.info(`Downloaded ${(totalBytes / 1024 / 1024).toFixed(2)} MB`);
                        resolve();
                    });
                }).on('error', (err) => {
                    fs.unlink(dest, () => { });
                    reject(err);
                }).end();
            };
            makeRequest(url);
        });
    }
    calculateChecksum(filePath) {
        return new Promise((resolve, reject) => {
            const hash = crypto.createHash('sha256');
            const stream = fs.createReadStream(filePath);
            stream.on('data', (data) => hash.update(data));
            stream.on('end', () => resolve(hash.digest('hex')));
            stream.on('error', reject);
        });
    }
    async extractPackage(packageFile, destDir) {
        if (!fs.existsSync(destDir))
            fs.mkdirSync(destDir, { recursive: true });
        if (packageFile.endsWith('.zip')) {
            if (process.platform === 'win32') {
                // Try multiple extraction methods on Windows for maximum compatibility
                const methods = [
                    // Method 1: tar (available on Windows 10 1803+)
                    `tar -xf "${packageFile}" -C "${destDir}"`,
                    // Method 2: .NET ZipFile (works on PowerShell 2.0+)
                    `powershell -NoProfile -Command "Add-Type -AssemblyName System.IO.Compression.FileSystem; [System.IO.Compression.ZipFile]::ExtractToDirectory('${packageFile}', '${destDir}')"`,
                    // Method 3: Expand-Archive (PowerShell 5.0+)
                    `powershell -NoProfile -Command "Expand-Archive -Path '${packageFile}' -DestinationPath '${destDir}' -Force"`,
                ];
                let lastErr = null;
                for (let i = 0; i < methods.length; i++) {
                    try {
                        logger_1.logger.info(`ZIP extraction attempt ${i + 1}/${methods.length}`);
                        await new Promise((resolve, reject) => {
                            (0, child_process_1.exec)(methods[i], { timeout: 180000 }, (err, stdout, stderr) => {
                                if (err) {
                                    logger_1.logger.warn(`Extraction method ${i + 1} failed: ${stderr || err.message}`);
                                    reject(err);
                                }
                                else {
                                    resolve();
                                }
                            });
                        });
                        logger_1.logger.info(`ZIP extraction succeeded with method ${i + 1}`);
                        return; // success
                    }
                    catch (err) {
                        lastErr = err;
                    }
                }
                throw lastErr || new Error('All ZIP extraction methods failed');
            }
            else {
                await new Promise((resolve, reject) => {
                    (0, child_process_1.exec)(`unzip -o "${packageFile}" -d "${destDir}"`, { timeout: 120000 }, (err) => {
                        if (err)
                            reject(err);
                        else
                            resolve();
                    });
                });
            }
        }
        else {
            // tar.gz
            await new Promise((resolve, reject) => {
                (0, child_process_1.exec)(`tar -xzf "${packageFile}" -C "${destDir}"`, { timeout: 120000 }, (err) => {
                    if (err)
                        reject(err);
                    else
                        resolve();
                });
            });
        }
    }
    async backupCurrent(backupDir) {
        if (!fs.existsSync(backupDir))
            fs.mkdirSync(backupDir, { recursive: true });
        const installPath = config_1.config.upgrade.installPath;
        const filesToBackup = ['package.json', 'dist', 'node_modules'];
        for (const file of filesToBackup) {
            const src = path.join(installPath, file);
            const dest = path.join(backupDir, file);
            if (fs.existsSync(src)) {
                try {
                    await this.copyRecursive(src, dest);
                }
                catch (e) {
                    logger_1.logger.warn(`Could not backup ${file}: ${e.message}`);
                }
            }
        }
        logger_1.logger.info(`Backed up to ${backupDir}`);
    }
    async replaceFiles(srcDir, destDir) {
        const entries = fs.readdirSync(srcDir, { withFileTypes: true });
        for (const entry of entries) {
            const src = path.join(srcDir, entry.name);
            const dest = path.join(destDir, entry.name);
            // Never overwrite config, .env, logs, backup, or node_modules
            if (['agent-config.json', '.env', 'logs', 'backup', 'updates', 'node_modules'].includes(entry.name))
                continue;
            if (entry.isDirectory()) {
                if (!fs.existsSync(dest))
                    fs.mkdirSync(dest, { recursive: true });
                await this.replaceFiles(src, dest);
            }
            else {
                fs.copyFileSync(src, dest);
            }
        }
    }
    async copyRecursive(src, dest) {
        const stat = fs.statSync(src);
        if (stat.isDirectory()) {
            if (!fs.existsSync(dest))
                fs.mkdirSync(dest, { recursive: true });
            const entries = fs.readdirSync(src);
            for (const entry of entries) {
                await this.copyRecursive(path.join(src, entry), path.join(dest, entry));
            }
        }
        else {
            fs.copyFileSync(src, dest);
        }
    }
    removeRecursive(dir) {
        if (fs.existsSync(dir)) {
            fs.readdirSync(dir, { withFileTypes: true }).forEach((entry) => {
                const full = path.join(dir, entry.name);
                if (entry.isDirectory())
                    this.removeRecursive(full);
                else
                    fs.unlinkSync(full);
            });
            fs.rmdirSync(dir);
        }
    }
    restartAgent(extractDir) {
        logger_1.logger.info('Restarting agent process...');
        const installPath = config_1.config.upgrade.installPath;
        // Build a file-copy command that runs AFTER the process stops
        // xcopy/robocopy on Windows, cp -r on Linux
        let copyCmd = '';
        let cleanCmd = '';
        if (extractDir) {
            if (process.platform === 'win32') {
                // robocopy copies directories; /E = recursive, /IS /IT = overwrite, /NFL /NDL /NJH /NJS = quiet
                copyCmd = `robocopy "${extractDir}" "${installPath}" /E /IS /IT /NFL /NDL /NJH /NJS /R:3 /W:2\nif errorlevel 8 echo [WARN] robocopy returned errorlevel %errorlevel%`;
                cleanCmd = `rmdir /s /q "${extractDir}" 2>nul`;
            }
            else {
                copyCmd = `cp -rf "${extractDir}"/* "${installPath}/"`;
                cleanCmd = `rm -rf "${extractDir}"`;
            }
        }
        if (process.platform === 'win32') {
            const isService = process.env.APPSTATS_SERVICE === '1' ||
                process.argv.some(a => a.includes('node-windows')) ||
                process.ppid === 1;
            if (isService) {
                const restartScript = path.join(installPath, 'restart-service.bat');
                fs.writeFileSync(restartScript, `@echo off
timeout /t 2 /nobreak >nul
net stop AppStatsAgent >nul 2>nul
timeout /t 3 /nobreak >nul
${copyCmd ? copyCmd + '\n' : ''}${cleanCmd ? cleanCmd + '\n' : ''}net start AppStatsAgent >nul 2>nul
if errorlevel 1 (
  timeout /t 3 /nobreak >nul
  net start AppStatsAgent >nul 2>nul
)
del "%~f0"
`);
                logger_1.logger.info('Restarting via Windows service (net stop/start)...');
                (0, child_process_1.spawn)('cmd', ['/c', restartScript], { detached: true, stdio: 'ignore' }).unref();
            }
            else {
                const restartScript = path.join(installPath, 'restart.bat');
                fs.writeFileSync(restartScript, `@echo off
timeout /t 2 /nobreak >nul
${copyCmd ? copyCmd + '\n' : ''}${cleanCmd ? cleanCmd + '\n' : ''}cd /d "${installPath}"
node dist\\index.js
del "%~f0"
`);
                (0, child_process_1.spawn)('cmd', ['/c', restartScript], { detached: true, stdio: 'ignore' }).unref();
            }
        }
        else {
            const restartScript = path.join(installPath, 'restart.sh');
            fs.writeFileSync(restartScript, `#!/bin/bash
sleep 2
${copyCmd ? copyCmd + '\n' : ''}${cleanCmd ? cleanCmd + '\n' : ''}cd "${installPath}"
node dist/index.js &
rm -f "$0"
`, { mode: 0o755 });
            (0, child_process_1.spawn)('bash', [restartScript], { detached: true, stdio: 'ignore' }).unref();
        }
        process.exit(0);
    }
    reportStatus(status, version, progress, error) {
        this.wsClient.send({
            type: 'upgrade:status',
            payload: {
                agentId: this.wsClient.getAgentId(),
                fromVersion: AGENT_VERSION,
                toVersion: version,
                status,
                progress,
                startedAt: new Date().toISOString(),
                error,
            },
        });
    }
    reportUpgradeResponse(response, version) {
        this.wsClient.send({
            type: 'upgrade:response',
            payload: {
                agentId: this.wsClient.getAgentId(),
                currentVersion: AGENT_VERSION,
                targetVersion: version,
                response,
                autoUpdate: config_1.config.upgrade.autoUpdate,
                timestamp: new Date().toISOString(),
            },
        });
    }
    getCurrentVersion() {
        return AGENT_VERSION;
    }
}
exports.UpgradeService = UpgradeService;
//# sourceMappingURL=upgrade.js.map