"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const config_1 = require("./config");
const logger_1 = require("./logger");
const client_1 = require("./websocket/client");
const system_1 = require("./collectors/system");
const process_1 = require("./collectors/process");
const productivity_1 = require("./collectors/productivity");
const network_1 = require("./collectors/network");
const system_events_1 = require("./collectors/system-events");
const software_1 = require("./collectors/software");
const login_sessions_1 = require("./collectors/login-sessions");
const browser_history_1 = require("./collectors/browser-history");
const probe_runner_1 = require("./probes/probe-runner");
const upgrade_1 = require("./services/upgrade");
const security_scanner_1 = require("./collectors/security-scanner");
const config_ui_1 = require("./config-ui");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const http_1 = __importDefault(require("http"));
const https_1 = __importDefault(require("https"));
const dns_1 = __importDefault(require("dns"));
const net_1 = __importDefault(require("net"));
const url_1 = require("url");
const VERSION = '1.4.9';
class AppStatsAgent {
    constructor() {
        this.intervals = [];
        this.probeTimers = new Map();
        this.probeInFlight = new Set();
        this.probeResultBuffers = new Map();
        this.probeFlushTimer = null;
        this.liveMonitoring = false;
        this.shuttingDown = false;
        this.configUIServer = null;
        this.logBuffer = [];
        this.speedtestInstallAttempted = false;
        this.speedtestLastInstallAttemptMs = 0;
        // WireGuard state — persisted across handler calls
        this.wgActive = false; // true once tunnel is up and WS switched to WG URL
        this.wgHealthMonitor = null;
        this.wgDeployUrl = ''; // original HTTP URL used to deploy the agent
        this.wgDeployWsUrl = ''; // original WS URL used to deploy the agent
        // ---- Remote Session Handlers ----
        this.activeSessions = new Map();
        /**
         * Describes what this agent did to a local service so it can be cleaned up
         * when the session ends.
         */
        this.remoteServiceState = new Map();
        this.wsClient = new client_1.AgentWSClient();
        this.systemCollector = new system_1.SystemCollector();
        this.processCollector = new process_1.ProcessCollector();
        this.productivityCollector = new productivity_1.ProductivityCollector();
        this.networkCollector = new network_1.NetworkCollector();
        this.systemEventsCollector = new system_events_1.SystemEventsCollector();
        this.softwareCollector = new software_1.SoftwareCollector();
        this.loginSessionCollector = new login_sessions_1.LoginSessionCollector();
        this.browserHistoryCollector = new browser_history_1.BrowserHistoryCollector();
        this.probeRunner = new probe_runner_1.ProbeRunner();
        this.upgradeService = new upgrade_1.UpgradeService(this.wsClient);
        this.securityScanner = new security_scanner_1.SecurityScanner();
    }
    async start() {
        logger_1.logger.info(`AppStats Agent v${VERSION} starting...`);
        logger_1.logger.info(`Hostname: ${config_1.config.agent.hostname}`);
        logger_1.logger.info(`Platform: ${config_1.config.agent.platform} ${config_1.config.agent.arch}`);
        logger_1.logger.info(`Server: ${config_1.config.server.wsUrl}`);
        // ---- Persist the original deploy URL (written once, never overwritten by WG switch) ----
        // This is the URL we always fall back to if the WireGuard tunnel fails.
        this.persistDeployUrl();
        // Always connect via deploy URL on startup, even if config.json has WG URLs.
        // The WG switch code will switch to the tunnel URL after confirming a handshake.
        if (this.wgDeployWsUrl && this.wgDeployWsUrl !== config_1.config.server.wsUrl) {
            logger_1.logger.info(`Startup: overriding WS URL from ${config_1.config.server.wsUrl} to deploy URL ${this.wgDeployWsUrl}`);
            config_1.config.server.url = this.wgDeployUrl;
            config_1.config.server.wsUrl = this.wgDeployWsUrl;
        }
        // Intercept logger to buffer log messages for the config UI
        this.setupLogCapture();
        this.setupEventHandlers();
        this.wsClient.connect();
        // Start the local configuration UI (accessible at http://localhost:9898)
        try {
            this.configUIServer = (0, config_ui_1.startConfigUI)({
                getStatus: () => this.getAgentStatus(),
                getConfig: () => this.getAgentConfig(),
                updateConfig: (updates) => this.updateAgentConfig(updates),
                restart: () => this.restartAgent(),
                getLogs: (lines) => this.getRecentLogs(lines),
            });
            logger_1.logger.info('Agent Config UI started on http://localhost:9898');
        }
        catch (e) {
            logger_1.logger.warn(`Could not start Config UI: ${e.message}`);
        }
        // Graceful shutdown
        process.on('SIGINT', () => this.shutdown());
        process.on('SIGTERM', () => this.shutdown());
        process.on('uncaughtException', (err) => {
            logger_1.logger.error('Uncaught exception:', err);
        });
        process.on('unhandledRejection', (err) => {
            logger_1.logger.error('Unhandled rejection:', err);
        });
    }
    // ---- Config UI Interface Methods ----
    setupLogCapture() {
        // Buffer last 500 log lines for the config UI
        const origInfo = logger_1.logger.info.bind(logger_1.logger);
        const origWarn = logger_1.logger.warn.bind(logger_1.logger);
        const origError = logger_1.logger.error.bind(logger_1.logger);
        const addLog = (level, msg) => {
            const line = `[${new Date().toISOString()}] [${level}] ${msg}`;
            this.logBuffer.push(line);
            if (this.logBuffer.length > 500)
                this.logBuffer.shift();
        };
        logger_1.logger.info = ((msg, ...args) => { addLog('INFO', msg); return origInfo(msg, ...args); });
        logger_1.logger.warn = ((msg, ...args) => { addLog('WARN', msg); return origWarn(msg, ...args); });
        logger_1.logger.error = ((msg, ...args) => { addLog('ERROR', msg); return origError(msg, ...args); });
    }
    getAgentStatus() {
        return {
            agentId: config_1.config.agent.hostname,
            version: VERSION,
            serverUrl: config_1.config.server.url,
            wsUrl: config_1.config.server.wsUrl,
            connected: this.wsClient.isConnected?.() || false,
            uptime: process.uptime(),
            pid: process.pid,
            platform: config_1.config.agent.platform,
            arch: config_1.config.agent.arch,
        };
    }
    getAgentConfig() {
        // Try to read the config file if it exists
        const configPath = process.env.APPSTATS_CONFIG || path_1.default.join(process.cwd(), 'agent-config.json');
        try {
            if (fs_1.default.existsSync(configPath)) {
                return JSON.parse(fs_1.default.readFileSync(configPath, 'utf-8'));
            }
        }
        catch { }
        // Return current running config
        return {
            serverUrl: config_1.config.server.url,
            wsUrl: config_1.config.server.wsUrl,
            token: config_1.config.server.token,
            tags: [],
            groups: [],
            collectors: {
                system: { enabled: true, interval: config_1.config.intervals.system },
                process: { enabled: true, interval: config_1.config.intervals.process },
                productivity: { enabled: true, interval: config_1.config.intervals.productivity },
                network: { enabled: true },
                activity: { enabled: true },
            },
            heartbeatInterval: 30000,
        };
    }
    updateAgentConfig(updates) {
        const configPath = process.env.APPSTATS_CONFIG || path_1.default.join(process.cwd(), 'agent-config.json');
        let existing = {};
        try {
            if (fs_1.default.existsSync(configPath)) {
                existing = JSON.parse(fs_1.default.readFileSync(configPath, 'utf-8'));
            }
        }
        catch { }
        const merged = { ...existing, ...updates };
        fs_1.default.writeFileSync(configPath, JSON.stringify(merged, null, 2), 'utf-8');
        logger_1.logger.info(`Configuration file updated at ${configPath}`);
    }
    restartAgent() {
        logger_1.logger.info('Agent restart requested via Config UI');
        setTimeout(() => {
            process.exit(0); // The service manager (systemd/node-windows) will restart it
        }, 1000);
    }
    getRecentLogs(lines) {
        const count = lines || 100;
        return this.logBuffer.slice(-count);
    }
    setupEventHandlers() {
        this.wsClient.on('registered', (agentId, serverConfig) => {
            logger_1.logger.info(`Registered with server as: ${agentId}`);
            this.startCollectors();
            this.applyServerConfig(serverConfig);
        });
        this.wsClient.on('config-update', (serverConfig) => {
            this.applyServerConfig(serverConfig);
        });
        this.wsClient.on('process-terminate', async (payload) => {
            const { pid, signal, name } = payload;
            logger_1.logger.info(`Terminate request: PID ${pid} (${name}) with ${signal}`);
            const success = await this.processCollector.terminate(pid, signal);
            this.wsClient.sendMetrics('process:terminate:result', {
                pid, name, signal, success,
                timestamp: new Date().toISOString(),
            });
        });
        this.wsClient.on('monitoring-toggle', (payload) => {
            this.liveMonitoring = payload.enabled;
            logger_1.logger.info(`Live monitoring ${this.liveMonitoring ? 'enabled' : 'disabled'}`);
            if (this.liveMonitoring) {
                this.startLiveMonitoring();
            }
        });
        this.wsClient.on('probe-start', (payload) => {
            this.startProbe(payload);
        });
        this.wsClient.on('probe-stop', (payload) => {
            this.stopProbe(payload.probeId);
        });
        this.wsClient.on('upgrade-command', (payload) => {
            if (payload?.action === 'shutdown') {
                logger_1.logger.warn(`Received server shutdown command: ${payload.reason || 'duplicate instance handling'}`);
                this.shutdown().finally(() => process.exit(0));
                return;
            }
            this.upgradeService.handleUpgradeCommand(payload);
        });
        this.wsClient.on('upgrade-available', (payload) => {
            this.upgradeService.handleUpgradeAvailable(payload);
        });
        this.wsClient.on('traffic-rules-update', (payload) => {
            this.networkCollector.updateTrafficRules(payload.rules || []);
        });
        this.wsClient.on('traffic-filters-update', (payload) => {
            this.networkCollector.updateURLFilters(payload.filters || []);
        });
        this.wsClient.on('request-system-info', async () => {
            const metrics = await this.systemCollector.collect();
            this.wsClient.sendMetrics('system:metrics', metrics);
        });
        this.wsClient.on('request-processes', async () => {
            const procs = await this.processCollector.collect();
            this.wsClient.sendMetrics('process:snapshot', {
                processes: procs.slice(0, 100),
                totalCount: procs.length,
            });
        });
        this.wsClient.on('request-software', async () => {
            await this.sendSoftwareSnapshot('server-request');
        });
        this.wsClient.on('software-uninstall', async (payload) => {
            await this.handleSoftwareUninstall(payload);
        });
        this.wsClient.on('extension-install', async (payload) => {
            this.installBrowserExtension(payload);
        });
        this.wsClient.on('security-scan', async (payload) => {
            this.handleSecurityScan(payload);
        });
        this.wsClient.on('remote-session-start', async (payload) => {
            this.handleRemoteSessionStart(payload);
        });
        this.wsClient.on('remote-session-end', async (payload) => {
            await this.handleRemoteSessionEnd(payload);
        });
        this.wsClient.on('wireguard-config', async (payload) => {
            this.handleWireGuardConfig(payload);
        });
        this.wsClient.on('speedtest-run', async (payload) => {
            this.handleSpeedtestRun(payload);
        });
        this.wsClient.on('disconnected', () => {
            logger_1.logger.warn('Lost connection to server');
        });
        this.wsClient.on('duplicate-instance', (info) => {
            logger_1.logger.warn(`Duplicate instance detected by server (${info?.code || ''} ${info?.reason || ''})`);
            this.shutdown().finally(() => process.exit(0));
        });
        this.wsClient.on('connected', () => {
            logger_1.logger.info('Reconnected to server');
            // Push buffered activity quickly after reconnect so dashboards repopulate.
            setTimeout(() => {
                if (!this.wsClient.isConnected())
                    return;
                const records = this.productivityCollector.flushRecords();
                if (records.length > 0) {
                    this.wsClient.send({
                        type: 'productivity:batch',
                        payload: records,
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
                const events = this.productivityCollector.flushEvents();
                if (events.length > 0) {
                    this.wsClient.send({
                        type: 'activity:batch',
                        payload: events,
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
                const screenshots = this.productivityCollector.flushScreenshots();
                if (screenshots.length > 0) {
                    this.wsClient.send({
                        type: 'productivity:screenshots',
                        payload: screenshots,
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
                this.sendSoftwareSnapshot('reconnect').catch(() => { });
            }, 2000);
        });
    }
    startCollectors() {
        // Clear existing intervals
        this.intervals.forEach(i => clearInterval(i));
        this.intervals = [];
        this.ensureSpeedtestCliInstalled().catch(() => { });
        // System metrics collection (5-second default)
        const systemInterval = setInterval(async () => {
            try {
                const metrics = await this.systemCollector.collect();
                this.wsClient.sendMetrics('system:metrics', metrics);
            }
            catch (e) {
                // Skipped - already collecting
            }
        }, config_1.config.intervals.system);
        this.intervals.push(systemInterval);
        // Agent self-monitoring (every 60 seconds)
        let prevCpuUsage = process.cpuUsage();
        let prevCpuTime = Date.now();
        const selfMonitorInterval = setInterval(() => {
            try {
                const memUsage = process.memoryUsage();
                const currentCpuUsage = process.cpuUsage(prevCpuUsage);
                const elapsedMs = Date.now() - prevCpuTime;
                const cpuPercent = ((currentCpuUsage.user + currentCpuUsage.system) / 1000) / (elapsedMs * 10); // % of one core
                prevCpuUsage = process.cpuUsage();
                prevCpuTime = Date.now();
                this.wsClient.sendMetrics('agent:self-monitor', {
                    pid: process.pid,
                    uptime: process.uptime(),
                    memory: {
                        rss: memUsage.rss,
                        heapTotal: memUsage.heapTotal,
                        heapUsed: memUsage.heapUsed,
                        external: memUsage.external,
                        rssMB: Math.round(memUsage.rss / 1024 / 1024 * 100) / 100,
                        heapUsedMB: Math.round(memUsage.heapUsed / 1024 / 1024 * 100) / 100,
                    },
                    cpu: {
                        user: currentCpuUsage.user,
                        system: currentCpuUsage.system,
                        percent: Math.round(cpuPercent * 100) / 100,
                    },
                    version: VERSION,
                    activeProbes: this.probeTimers.size,
                    wsConnected: this.wsClient.isConnected?.() || false,
                });
            }
            catch (e) {
                // Self-monitoring is non-critical
            }
        }, 60000);
        this.intervals.push(selfMonitorInterval);
        // Process collection (10-second default)
        const processInterval = setInterval(async () => {
            try {
                const procs = await this.processCollector.collect();
                this.wsClient.sendMetrics('process:snapshot', {
                    processes: procs.slice(0, 100), // Top 100 processes
                    totalCount: procs.length,
                });
            }
            catch (e) {
                logger_1.logger.error('Process collection error:', e);
            }
        }, config_1.config.intervals.process);
        this.intervals.push(processInterval);
        // Productivity tracking (3-second default)
        const prodInterval = setInterval(async () => {
            await this.productivityCollector.collectActiveWindow();
        }, config_1.config.intervals.productivity);
        this.intervals.push(prodInterval);
        // Flush productivity records every 30 seconds
        const prodFlushInterval = setInterval(() => {
            if (!this.wsClient.isConnected())
                return;
            const records = this.productivityCollector.flushRecords();
            if (records.length > 0) {
                this.wsClient.send({
                    type: 'productivity:batch',
                    payload: records,
                    agentId: this.wsClient.getAgentId() || undefined,
                });
            }
            const events = this.productivityCollector.flushEvents();
            if (events.length > 0) {
                this.wsClient.send({
                    type: 'activity:batch',
                    payload: events,
                    agentId: this.wsClient.getAgentId() || undefined,
                });
            }
            const screenshots = this.productivityCollector.flushScreenshots();
            if (screenshots.length > 0) {
                this.wsClient.send({
                    type: 'productivity:screenshots',
                    payload: screenshots,
                    agentId: this.wsClient.getAgentId() || undefined,
                });
            }
        }, 30000);
        this.intervals.push(prodFlushInterval);
        // Network stats (30-second interval)
        const networkInterval = setInterval(async () => {
            try {
                const connections = await this.networkCollector.collectConnections();
                const stats = await this.networkCollector.collectStats();
                // Evaluate connections against rules
                const evaluated = connections.map(conn => ({
                    ...conn,
                    ...this.networkCollector.evaluateConnection(conn),
                }));
                const blocked = evaluated.filter(c => c.action === 'block' || c.action === 'log');
                if (blocked.length > 0) {
                    this.wsClient.sendMetrics('traffic:log', { entries: blocked });
                }
                // Send network stats periodically
                this.wsClient.sendMetrics('network:stats', { stats, connectionCount: connections.length });
            }
            catch (e) {
                logger_1.logger.error('Network collection error:', e);
            }
        }, 30000);
        this.intervals.push(networkInterval);
        // System events (every 5 minutes)
        const sysEventsInterval = setInterval(async () => {
            try {
                const events = await this.systemEventsCollector.collect();
                if (events.length > 0) {
                    this.wsClient.send({
                        type: 'system:events',
                        payload: events,
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
            }
            catch (e) {
                logger_1.logger.error('System events collection error:', e);
            }
        }, 300000); // 5 minutes
        this.intervals.push(sysEventsInterval);
        // Collect system events once on startup
        setTimeout(async () => {
            try {
                const events = await this.systemEventsCollector.collect();
                if (events.length > 0) {
                    this.wsClient.send({
                        type: 'system:events',
                        payload: events,
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
            }
            catch (e) {
                logger_1.logger.debug('Initial system events collection failed');
            }
        }, 15000);
        // Installed software (every 5 minutes for faster recovery if startup send fails)
        const softwareInterval = setInterval(async () => {
            await this.sendSoftwareSnapshot('interval');
        }, 5 * 60000); // 5 minutes
        this.intervals.push(softwareInterval);
        // Collect software once on startup
        setTimeout(async () => {
            await this.sendSoftwareSnapshot('startup');
        }, 20000);
        // Login sessions (every 2 minutes)
        const loginInterval = setInterval(async () => {
            try {
                const sessions = await this.loginSessionCollector.collect();
                if (sessions.length > 0) {
                    this.wsClient.send({
                        type: 'session:batch',
                        payload: { sessions },
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
            }
            catch (e) {
                logger_1.logger.error('Login session collection error:', e);
            }
        }, 120000); // 2 minutes
        this.intervals.push(loginInterval);
        // Collect login sessions once on startup
        setTimeout(async () => {
            try {
                const sessions = await this.loginSessionCollector.collect();
                if (sessions.length > 0) {
                    this.wsClient.send({
                        type: 'session:batch',
                        payload: { sessions },
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
            }
            catch (e) {
                logger_1.logger.debug('Initial login session collection failed');
            }
        }, 10000);
        // Browser history collection (every 2 minutes)
        const browserInterval = setInterval(async () => {
            try {
                const entries = await this.browserHistoryCollector.collect();
                if (entries.length > 0) {
                    const visits = entries.map(e => ({
                        agentId: this.wsClient.getAgentId() || config_1.config.agent.hostname,
                        timestamp: e.visitTime,
                        url: e.url,
                        title: e.title,
                        domain: e.domain,
                        duration: e.visitDuration,
                        browser: e.browser,
                        tabId: 0,
                        isActive: true,
                    }));
                    this.wsClient.send({
                        type: 'browser:batch',
                        payload: { visits },
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
            }
            catch (e) {
                logger_1.logger.error('Browser history collection error:', e);
            }
        }, 120000); // 2 minutes
        this.intervals.push(browserInterval);
        // Collect browser history once on startup (delayed)
        setTimeout(async () => {
            try {
                const entries = await this.browserHistoryCollector.collect();
                if (entries.length > 0) {
                    const visits = entries.map(e => ({
                        agentId: this.wsClient.getAgentId() || config_1.config.agent.hostname,
                        timestamp: e.visitTime,
                        url: e.url,
                        title: e.title,
                        domain: e.domain,
                        duration: e.visitDuration,
                        browser: e.browser,
                        tabId: 0,
                        isActive: true,
                    }));
                    this.wsClient.send({
                        type: 'browser:batch',
                        payload: { visits },
                        agentId: this.wsClient.getAgentId() || undefined,
                    });
                }
            }
            catch (e) {
                logger_1.logger.debug('Initial browser history collection failed');
            }
        }, 25000);
        // WiFi metrics and IP address reporting (every 60 seconds)
        const wifiInterval = setInterval(async () => {
            try {
                const wifiInfo = await this.systemCollector.collectWifi();
                const ipInfo = await this.systemCollector.collectIPAddresses();
                if (wifiInfo) {
                    this.wsClient.sendMetrics('network:wifi', {
                        ...wifiInfo,
                        ip_addresses: ipInfo.all,
                        timestamp: new Date().toISOString(),
                    });
                }
                // Always send IP update
                if (ipInfo.all.length > 0) {
                    this.wsClient.sendMetrics('agent:ip-update', {
                        ip_addresses: ipInfo.all,
                        primary_ip: ipInfo.primary,
                    });
                }
            }
            catch (e) {
                logger_1.logger.debug('WiFi/IP collection error:', e);
            }
        }, 60000);
        this.intervals.push(wifiInterval);
        // Collect wifi/IP once on startup (delayed)
        setTimeout(async () => {
            try {
                const ipInfo = await this.systemCollector.collectIPAddresses();
                if (ipInfo.all.length > 0) {
                    this.wsClient.sendMetrics('agent:ip-update', {
                        ip_addresses: ipInfo.all,
                        primary_ip: ipInfo.primary,
                    });
                }
                const wifiInfo = await this.systemCollector.collectWifi();
                if (wifiInfo) {
                    this.wsClient.sendMetrics('network:wifi', {
                        ...wifiInfo,
                        ip_addresses: ipInfo.all,
                        timestamp: new Date().toISOString(),
                    });
                }
            }
            catch (e) {
                logger_1.logger.debug('Initial WiFi/IP collection failed');
            }
        }, 15000);
        logger_1.logger.info('All collectors started');
    }
    async sendSoftwareSnapshot(context) {
        if (!this.wsClient.isConnected()) {
            logger_1.logger.debug(`Skipping software collection (${context}) while disconnected`);
            return;
        }
        try {
            const software = await this.softwareCollector.collect();
            if (software.length > 0) {
                this.wsClient.send({
                    type: 'software:list',
                    payload: { software },
                    agentId: this.wsClient.getAgentId() || undefined,
                });
            }
            else {
                logger_1.logger.debug(`Software collection returned no entries (${context})`);
            }
        }
        catch (e) {
            logger_1.logger.error('Software collection error:', e);
        }
    }
    async handleSoftwareUninstall(payload) {
        const agentId = this.wsClient.getAgentId() || config_1.config.agent.hostname;
        const requestedBy = String(payload?.requestedBy || 'unknown').trim() || 'unknown';
        const softwareName = String(payload?.name || '').trim();
        const sendResult = (success, message, extras) => {
            this.wsClient.sendMetrics('software:uninstall:result', {
                agentId,
                requestedBy,
                name: softwareName,
                success,
                message,
                ...(extras || {}),
                timestamp: new Date().toISOString(),
            });
        };
        if (!softwareName) {
            sendResult(false, 'Software name is required');
            return;
        }
        if (process.platform !== 'win32') {
            sendResult(false, `Software uninstall is not supported on ${process.platform} agents yet`);
            return;
        }
        const escapedSoftwareName = softwareName.replace(/'/g, "''");
        const script = `
$ErrorActionPreference = 'Stop'
$ProgressPreference = 'SilentlyContinue'
$target = '${escapedSoftwareName}'
$paths = @(
  'HKLM:\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Uninstall\\*',
  'HKLM:\\SOFTWARE\\WOW6432Node\\Microsoft\\Windows\\CurrentVersion\\Uninstall\\*',
  'HKCU:\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Uninstall\\*'
)
try {
  $entries = @()
  foreach ($regPath in $paths) {
    try {
      $entries += Get-ItemProperty $regPath -ErrorAction SilentlyContinue | Where-Object {
        $_.DisplayName -and (
          $_.DisplayName.Trim().ToLowerInvariant() -eq $target.Trim().ToLowerInvariant() -or
          $_.DisplayName.Trim().ToLowerInvariant().Contains($target.Trim().ToLowerInvariant())
        )
      }
    } catch {}
  }
  $entries = @($entries | Sort-Object PSPath -Unique)
  if ($entries.Count -eq 0) {
    throw "No installed software matched '$target'"
  }
  $messages = @()
  $successCount = 0
  foreach ($entry in $entries) {
    $displayName = [string]$entry.DisplayName
    $uninstallCmd = [string]$entry.QuietUninstallString
    if (-not $uninstallCmd) { $uninstallCmd = [string]$entry.UninstallString }
    if (-not $uninstallCmd) {
      $messages += "\${displayName}: uninstall command not found"
      continue
    }
    $invokeCmd = $uninstallCmd
    if ($invokeCmd -match '(?i)msiexec(\\.exe)?') {
      $invokeCmd = [regex]::Replace($invokeCmd, '(?i)/I(\\s+)', '/X$1')
      if ($invokeCmd -notmatch '(?i)\\s/q[nrb]') { $invokeCmd += ' /qn /norestart' }
    } elseif ($invokeCmd -notmatch '(?i)\\s/(quiet|silent|s|verysilent)\\b') {
      $invokeCmd += ' /quiet'
    }
    try {
      $proc = Start-Process -FilePath 'cmd.exe' -ArgumentList '/c', $invokeCmd -Wait -PassThru -WindowStyle Hidden
      $exitCode = [int]($proc.ExitCode)
      if ($exitCode -eq 0 -or $exitCode -eq 1641 -or $exitCode -eq 3010) {
        $successCount++
        $messages += "\${displayName}: success ($exitCode)"
      } else {
        $messages += "\${displayName}: exit code $exitCode"
      }
    } catch {
      $messages += "\${displayName}: $($_.Exception.Message)"
    }
  }
  if ($successCount -lt 1) {
    throw ($messages -join '; ')
  }
  @{
    success = $true
    removed = $successCount
    total = $entries.Count
    message = ($messages -join '; ')
  } | ConvertTo-Json -Compress
} catch {
  @{
    success = $false
    removed = 0
    total = 0
    message = [string]$_.Exception.Message
  } | ConvertTo-Json -Compress
}
exit 0
`;
        try {
            const output = await this.runPowerShellScript(script, 10 * 60 * 1000);
            const parsed = this.parseJsonOutput(output) || {};
            if (parsed && parsed.success === false) {
                sendResult(false, String(parsed.message || `Failed to uninstall ${softwareName}`), {
                    removed: 0,
                    total: Number(parsed.total || 0),
                });
                return;
            }
            const removed = Number(parsed.removed || 0);
            const total = Number(parsed.total || 0);
            sendResult(true, String(parsed.message || `Uninstall command executed for ${softwareName}`), {
                removed: Number.isFinite(removed) ? removed : 0,
                total: Number.isFinite(total) ? total : 0,
            });
            await this.sendSoftwareSnapshot('post-uninstall');
        }
        catch (error) {
            sendResult(false, String(error?.message || `Failed to uninstall ${softwareName}`));
        }
    }
    startLiveMonitoring() {
        // When live mode is enabled, send system metrics more frequently (1 second)
        const liveInterval = setInterval(async () => {
            if (!this.liveMonitoring) {
                clearInterval(liveInterval);
                return;
            }
            try {
                const quickMetrics = await this.systemCollector.collectQuick();
                this.wsClient.sendMetrics('system:live', quickMetrics);
            }
            catch { }
        }, 1000);
        this.intervals.push(liveInterval);
    }
    applyServerConfig(serverConfig) {
        if (!serverConfig)
            return;
        // Apply probe configurations
        if (serverConfig.probes) {
            for (const probe of serverConfig.probes) {
                if (probe.enabled) {
                    this.startProbe(probe);
                }
            }
        }
        // Apply traffic rules
        if (serverConfig.trafficRules) {
            this.networkCollector.updateTrafficRules(serverConfig.trafficRules);
        }
        // Apply URL filters
        if (serverConfig.urlFilters) {
            this.networkCollector.updateURLFilters(serverConfig.urlFilters);
        }
        // Apply productivity rules
        if (serverConfig.productivityRules) {
            const rulesMap = {};
            const normalizeScore = (value) => {
                const raw = String(value ?? '').trim().toLowerCase();
                if (raw === 'productive' || raw === 'neutral' || raw === 'unproductive')
                    return raw;
                const n = Number(raw);
                if (Number.isFinite(n)) {
                    if (n >= 7)
                        return 'productive';
                    if (n <= 3)
                        return 'unproductive';
                }
                return 'neutral';
            };
            for (const rule of serverConfig.productivityRules) {
                const pattern = String(rule.pattern || '').trim().toLowerCase();
                if (!pattern)
                    continue;
                rulesMap[pattern] = normalizeScore(rule.score);
            }
            this.productivityCollector.updateRules(rulesMap);
        }
        // Apply granular tracking config from tenant settings
        if (serverConfig.trackingConfig || serverConfig.tracking_mode !== undefined) {
            const tc = serverConfig.trackingConfig || serverConfig;
            this.productivityCollector.applyTrackingConfig({
                tracking_mode: tc.tracking_mode,
                track_active_window: tc.track_active_window,
                track_keystrokes_activity: tc.track_keystrokes_activity,
                track_keystrokes_log: tc.track_keystrokes_log,
                track_mouse_activity: tc.track_mouse_activity,
                track_browser_urls: tc.track_browser_urls,
                track_browser_titles: tc.track_browser_titles,
                track_network_traffic: tc.track_network_traffic,
                track_screenshots: tc.track_screenshots,
                screenshot_interval_seconds: tc.screenshot_interval_seconds,
                privacy_anon_users: tc.privacy_anon_users,
                privacy_blur_titles: tc.privacy_blur_titles,
                privacy_exclude_urls: tc.privacy_exclude_urls,
            });
        }
        logger_1.logger.info('Server configuration applied');
    }
    startProbe(probeConfig) {
        const { id, type, target, interval, count, timeout } = probeConfig;
        const probeId = String(id || '');
        if (!probeId || !type || !target) {
            logger_1.logger.warn('Invalid probe configuration received', { probeConfig });
            return;
        }
        // Stop existing probe timer
        this.stopProbe(probeId);
        const intervalSeconds = Math.max(1, Number(interval) || 300);
        logger_1.logger.info(`Starting probe: ${type} -> ${target} every ${intervalSeconds}s`);
        const runProbe = async () => {
            if (this.probeInFlight.has(probeId)) {
                logger_1.logger.debug(`Probe ${probeId} still running, skipping overlap`);
                return;
            }
            this.probeInFlight.add(probeId);
            try {
                let resultType = '';
                let result = null;
                if (type === 'ping') {
                    // Use one packet per scheduled run so 1s probes can produce 1 record per second.
                    result = await this.probeRunner.runPing(target, 1, timeout || 5000);
                    resultType = 'probe:ping-result';
                }
                else if (type === 'mtr' || type === 'traceroute') {
                    result = await this.probeRunner.runMTR(target, count || 10, timeout || 30000);
                    resultType = 'probe:mtr-result';
                }
                else if (type === 'http') {
                    result = await this.runHTTPProbe(target, timeout || 10000);
                    resultType = 'probe:http-result';
                }
                else if (type === 'dns') {
                    result = await this.runDNSProbe(target, timeout || 5000);
                    resultType = 'probe:dns-result';
                }
                else if (type === 'tcp') {
                    result = await this.runTCPProbe(target, timeout || 5000);
                    resultType = 'probe:tcp-result';
                }
                else {
                    logger_1.logger.warn(`Unknown probe type: ${type}`);
                }
                if (result && resultType) {
                    // Buffer results for batched sending
                    const bufKey = `${probeId}:${resultType}`;
                    if (!this.probeResultBuffers.has(bufKey)) {
                        this.probeResultBuffers.set(bufKey, { type: resultType, probeId: probeId, results: [] });
                    }
                    this.probeResultBuffers.get(bufKey).results.push({
                        ...result,
                        timestamp: new Date().toISOString(),
                    });
                    // Start flush timer if not already running
                    if (!this.probeFlushTimer) {
                        this.probeFlushTimer = setInterval(() => this.flushProbeResults(), 30000);
                    }
                }
            }
            catch (e) {
                logger_1.logger.error(`Probe ${probeId} error:`, e);
                this.wsClient.sendMetrics('probe:error', { probeId: probeId, error: e.message });
            }
            finally {
                this.probeInFlight.delete(probeId);
            }
        };
        // Run immediately
        runProbe();
        // Schedule recurring
        const timer = setInterval(runProbe, intervalSeconds * 1000);
        this.probeTimers.set(probeId, timer);
    }
    async runHTTPProbe(target, timeout) {
        return new Promise((resolve, reject) => {
            const startTime = Date.now();
            let ttfb = 0;
            try {
                const url = new url_1.URL(target.startsWith('http') ? target : `https://${target}`);
                const client = url.protocol === 'https:' ? https_1.default : http_1.default;
                const req = client.request(url, { method: 'GET', timeout }, (res) => {
                    ttfb = Date.now() - startTime;
                    let body = '';
                    let dataSize = 0;
                    res.on('data', (chunk) => {
                        dataSize += chunk.length;
                        if (body.length < 1024)
                            body += chunk.toString();
                    });
                    res.on('end', () => {
                        const totalTime = Date.now() - startTime;
                        const tlsInfo = res.socket.getPeerCertificate ? {
                            valid: !res.socket.authorizationError,
                            issuer: (res.socket.getPeerCertificate?.() || {}).issuer?.O || 'unknown',
                            expiresAt: (res.socket.getPeerCertificate?.() || {}).valid_to || null
                        } : null;
                        resolve({
                            url: target,
                            statusCode: res.statusCode,
                            statusMessage: res.statusMessage,
                            headers: res.headers,
                            ttfbMs: ttfb,
                            totalTimeMs: totalTime,
                            responseSizeBytes: dataSize,
                            contentType: res.headers['content-type'] || null,
                            tls: tlsInfo,
                            timestamp: new Date().toISOString()
                        });
                    });
                });
                req.on('error', (err) => {
                    resolve({
                        url: target,
                        error: err.message,
                        totalTimeMs: Date.now() - startTime,
                        ttfbMs: ttfb || null,
                        timestamp: new Date().toISOString()
                    });
                });
                req.on('timeout', () => {
                    req.destroy();
                    resolve({
                        url: target,
                        error: 'Request timed out',
                        totalTimeMs: Date.now() - startTime,
                        ttfbMs: ttfb || null,
                        timestamp: new Date().toISOString()
                    });
                });
                req.end();
            }
            catch (err) {
                resolve({
                    url: target,
                    error: err.message,
                    totalTimeMs: Date.now() - startTime,
                    timestamp: new Date().toISOString()
                });
            }
        });
    }
    async runDNSProbe(target, timeout) {
        return new Promise((resolve) => {
            const startTime = Date.now();
            const hostname = target.replace(/^https?:\/\//, '').split('/')[0].split(':')[0];
            const timer = setTimeout(() => {
                resolve({
                    hostname,
                    error: 'DNS lookup timed out',
                    totalTimeMs: Date.now() - startTime,
                    timestamp: new Date().toISOString()
                });
            }, timeout);
            const results = { hostname, timestamp: new Date().toISOString(), records: {} };
            // Resolve A records
            dns_1.default.resolve4(hostname, (err, addresses) => {
                if (!err)
                    results.records.A = addresses;
                else
                    results.records.A_error = err.message;
                // Resolve AAAA records
                dns_1.default.resolve6(hostname, (err6, addresses6) => {
                    if (!err6)
                        results.records.AAAA = addresses6;
                    else
                        results.records.AAAA_error = err6?.message;
                    // Resolve MX records
                    dns_1.default.resolveMx(hostname, (errMx, mx) => {
                        if (!errMx)
                            results.records.MX = mx;
                        else
                            results.records.MX_error = errMx?.message;
                        // Resolve NS records
                        dns_1.default.resolveNs(hostname, (errNs, ns) => {
                            if (!errNs)
                                results.records.NS = ns;
                            else
                                results.records.NS_error = errNs?.message;
                            clearTimeout(timer);
                            results.totalTimeMs = Date.now() - startTime;
                            resolve(results);
                        });
                    });
                });
            });
        });
    }
    async runTCPProbe(target, timeout) {
        return new Promise((resolve) => {
            const startTime = Date.now();
            // Parse host:port, default port 80
            const parts = target.replace(/^https?:\/\//, '').split('/')[0].split(':');
            const host = parts[0];
            const port = parseInt(parts[1]) || 80;
            const socket = new net_1.default.Socket();
            let banner = '';
            let connected = false;
            let connectTime = 0;
            socket.setTimeout(timeout);
            socket.on('connect', () => {
                connectTime = Date.now() - startTime;
                connected = true;
            });
            socket.on('data', (data) => {
                banner += data.toString().substring(0, 512);
                // Got a banner, close
                socket.end();
            });
            socket.on('timeout', () => {
                socket.destroy();
                resolve({
                    host, port, connected,
                    error: connected ? 'Read timed out after connect' : 'Connection timed out',
                    connectTimeMs: connectTime || null,
                    totalTimeMs: Date.now() - startTime,
                    timestamp: new Date().toISOString()
                });
            });
            socket.on('error', (err) => {
                resolve({
                    host, port, connected,
                    error: err.message,
                    connectTimeMs: connectTime || null,
                    totalTimeMs: Date.now() - startTime,
                    timestamp: new Date().toISOString()
                });
            });
            socket.on('end', () => {
                resolve({
                    host, port, connected: true,
                    connectTimeMs: connectTime,
                    totalTimeMs: Date.now() - startTime,
                    banner: banner || null,
                    timestamp: new Date().toISOString()
                });
            });
            // If no data after 2s post-connect, resolve anyway
            socket.on('connect', () => {
                setTimeout(() => {
                    if (socket.writable) {
                        socket.end();
                    }
                }, 2000);
            });
            socket.connect(port, host);
        });
    }
    stopProbe(probeId) {
        const timer = this.probeTimers.get(probeId);
        if (timer) {
            clearInterval(timer);
            this.probeTimers.delete(probeId);
            logger_1.logger.info(`Stopped probe: ${probeId}`);
        }
        this.probeInFlight.delete(probeId);
        for (const [bufKey, buf] of this.probeResultBuffers.entries()) {
            if (buf.probeId === probeId || bufKey.startsWith(`${probeId}:`)) {
                this.probeResultBuffers.delete(bufKey);
            }
        }
        if (this.probeTimers.size === 0 && this.probeFlushTimer) {
            clearInterval(this.probeFlushTimer);
            this.probeFlushTimer = null;
        }
    }
    runShellCommand(command, timeoutMs = 180000) {
        return new Promise((resolve, reject) => {
            const { exec } = require('child_process');
            exec(command, {
                timeout: timeoutMs,
                maxBuffer: 8 * 1024 * 1024,
                windowsHide: true,
            }, (error, stdout, stderr) => {
                if (error) {
                    reject(new Error((stderr || error.message || '').toString().trim() || 'Command failed'));
                    return;
                }
                resolve(String(stdout || stderr || '').trim());
            });
        });
    }
    runPowerShellScript(script, timeoutMs = 180000) {
        const encoded = Buffer.from(String(script || ''), 'utf16le').toString('base64');
        return this.runShellCommand(`powershell -NoProfile -NonInteractive -ExecutionPolicy Bypass -EncodedCommand ${encoded}`, timeoutMs);
    }
    parseJsonOutput(raw) {
        const text = String(raw || '').trim();
        if (!text)
            return null;
        try {
            return JSON.parse(text);
        }
        catch { }
        for (let i = text.indexOf('{'); i >= 0; i = text.indexOf('{', i + 1)) {
            for (let j = text.lastIndexOf('}'); j > i; j = text.lastIndexOf('}', j - 1)) {
                try {
                    return JSON.parse(text.slice(i, j + 1));
                }
                catch {
                    // continue scanning candidates
                }
            }
        }
        return null;
    }
    parseSpeedtestTextOutput(raw) {
        const text = String(raw || '').trim();
        if (!text)
            return null;
        const pick = (re) => {
            const m = text.match(re);
            return m?.[1] ? String(m[1]).trim() : '';
        };
        const toNum = (value) => {
            const n = Number(String(value || '').replace(/,/g, '.'));
            return Number.isFinite(n) ? n : 0;
        };
        const download = toNum(pick(/download(?:\s*speed)?\s*[:=]\s*([\d.,]+)/i));
        const upload = toNum(pick(/upload(?:\s*speed)?\s*[:=]\s*([\d.,]+)/i));
        const ping = toNum(pick(/(?:^|\s)(?:ping|latency)\s*[:=]\s*([\d.,]+)/im));
        const jitterRaw = pick(/jitter\s*[:=]\s*([\d.,]+)/i);
        const jitter = jitterRaw ? toNum(jitterRaw) : null;
        const serverFromHosted = pick(/hosted by\s+(.+?)(?:\s+\[[^\]]+\])?(?::\s*[\d.,]+\s*ms)?$/im);
        const serverFromLine = pick(/server\s*[:=]\s*(.+)$/im);
        const serverName = serverFromHosted || serverFromLine;
        const locationBits = [];
        const hostedLoc = pick(/hosted by\s+.*?\(([^)]+)\)/im);
        if (hostedLoc)
            locationBits.push(hostedLoc);
        const serverLoc = pick(/server\s*[:=]\s*.*?\(([^)]+)\)/im);
        if (serverLoc && !locationBits.includes(serverLoc))
            locationBits.push(serverLoc);
        const sponsor = serverFromHosted || '';
        const resultUrl = pick(/result\s*url\s*[:=]\s*(https?:\/\/\S+)/im);
        const shareUrl = pick(/(https?:\/\/\S+\.(?:png|jpg|jpeg))/im);
        const imageUrl = resultUrl || shareUrl || null;
        if (!(download > 0 || upload > 0 || ping > 0))
            return null;
        return {
            downloadMbps: download,
            uploadMbps: upload,
            pingMs: ping,
            jitterMs: jitter,
            serverName,
            serverLocation: locationBits.join(', '),
            sponsor,
            imageUrl,
        };
    }
    getSpeedtestChecks() {
        const runtimeNode = String(process.execPath || '').replace(/"/g, '\\"');
        const localSpeedtestCmd = path_1.default.resolve(__dirname, '../node_modules/.bin/speedtest-cli.cmd').replace(/\//g, '\\');
        const localSpeedtestCliJs = path_1.default.resolve(__dirname, '../node_modules/speedtest-cli/bin/cli.js').replace(/\//g, '\\');
        return [
            'cmd /c where speedtest-cli',
            'cmd /c where speedtest',
            `cmd /c if exist "${localSpeedtestCmd}" ("${localSpeedtestCmd}" --version) else exit /b 1`,
            runtimeNode
                ? `cmd /c if exist "${localSpeedtestCliJs}" ("${runtimeNode}" "${localSpeedtestCliJs}" --version) else exit /b 1`
                : '',
            'cmd /c if exist "%ProgramFiles%\\Ookla\\Speedtest\\speedtest.exe" (echo ok) else exit /b 1',
            'cmd /c if exist "%ProgramFiles(x86)%\\Ookla\\Speedtest\\speedtest.exe" (echo ok) else exit /b 1',
            'cmd /c if exist "%AppData%\\npm\\speedtest-cli.cmd" (echo ok) else exit /b 1',
            'cmd /c if exist "%AppData%\\npm\\speedtest.cmd" (echo ok) else exit /b 1',
            'cmd /c if exist "%ProgramData%\\AppStats\\npm\\speedtest-cli.cmd" (echo ok) else exit /b 1',
            'cmd /c if exist "%ProgramData%\\AppStats\\npm\\speedtest.cmd" (echo ok) else exit /b 1',
            'cmd /c if exist "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" (echo ok) else exit /b 1',
            'cmd /c if exist "%ProgramFiles%\\nodejs\\node.exe" if exist "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" ("%ProgramFiles%\\nodejs\\node.exe" "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" --version) else exit /b 1',
            'cmd /c if exist "%ProgramFiles(x86)%\\nodejs\\node.exe" if exist "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" ("%ProgramFiles(x86)%\\nodejs\\node.exe" "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" --version) else exit /b 1',
            runtimeNode
                ? `cmd /c if exist "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" ("${runtimeNode}" "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" --version) else exit /b 1`
                : '',
            'cmd /c npx --yes speedtest-cli --version',
            'cmd /c py -m speedtest --version',
            'cmd /c py -3 -m speedtest --version',
            'cmd /c python -m speedtest --version',
            'cmd /c python3 -m speedtest --version',
        ].filter((cmd) => !!String(cmd).trim());
    }
    getSpeedtestInstallAttempts() {
        return [
            'cmd /c py -m ensurepip --default-pip',
            'cmd /c python -m ensurepip --default-pip',
            'cmd /c python3 -m ensurepip --default-pip',
            'cmd /c npm config set fund false',
            'cmd /c npm config set audit false',
            'cmd /c if exist "%ProgramFiles%\\nodejs\\npm.cmd" ("%ProgramFiles%\\nodejs\\npm.cmd" config set fund false) else exit /b 1',
            'cmd /c if exist "%ProgramFiles%\\nodejs\\npm.cmd" ("%ProgramFiles%\\nodejs\\npm.cmd" config set audit false) else exit /b 1',
            'cmd /c if exist "%ProgramFiles(x86)%\\nodejs\\npm.cmd" ("%ProgramFiles(x86)%\\nodejs\\npm.cmd" config set fund false) else exit /b 1',
            'cmd /c if exist "%ProgramFiles(x86)%\\nodejs\\npm.cmd" ("%ProgramFiles(x86)%\\nodejs\\npm.cmd" config set audit false) else exit /b 1',
            'cmd /c pip install speedtest-cli',
            'cmd /c pip install --user speedtest-cli',
            'cmd /c py -m pip install speedtest-cli',
            'cmd /c py -m pip install --user speedtest-cli',
            'cmd /c python -m pip install speedtest-cli',
            'cmd /c python -m pip install --user speedtest-cli',
            'cmd /c python3 -m pip install speedtest-cli',
            'cmd /c python3 -m pip install --user speedtest-cli',
            'cmd /c npm install -g speedtest-cli --silent --no-progress',
            'cmd /c npm install --location=global speedtest-cli --silent --no-progress',
            'cmd /c if exist "%ProgramFiles%\\nodejs\\npm.cmd" ("%ProgramFiles%\\nodejs\\npm.cmd" install -g speedtest-cli --silent --no-progress) else exit /b 1',
            'cmd /c if exist "%ProgramFiles(x86)%\\nodejs\\npm.cmd" ("%ProgramFiles(x86)%\\nodejs\\npm.cmd" install -g speedtest-cli --silent --no-progress) else exit /b 1',
            'cmd /c if not exist "%ProgramData%\\AppStats\\npm" mkdir "%ProgramData%\\AppStats\\npm"',
            'cmd /c npm install -g speedtest-cli --prefix "%ProgramData%\\AppStats\\npm" --silent --no-progress',
            'cmd /c if exist "%ProgramFiles%\\nodejs\\npm.cmd" ("%ProgramFiles%\\nodejs\\npm.cmd" install -g speedtest-cli --prefix "%ProgramData%\\AppStats\\npm" --silent --no-progress) else exit /b 1',
            'cmd /c if exist "%ProgramFiles(x86)%\\nodejs\\npm.cmd" ("%ProgramFiles(x86)%\\nodejs\\npm.cmd" install -g speedtest-cli --prefix "%ProgramData%\\AppStats\\npm" --silent --no-progress) else exit /b 1',
            'cmd /c winget install --id Ookla.Speedtest.CLI -e --accept-source-agreements --accept-package-agreements --silent',
            'cmd /c choco install speedtest -y',
        ];
    }
    async ensureSpeedtestCliInstalled() {
        if (process.platform !== 'win32')
            return;
        const checks = this.getSpeedtestChecks();
        for (const check of checks) {
            try {
                await this.runShellCommand(check, 10000);
                return;
            }
            catch {
                // continue
            }
        }
        const now = Date.now();
        const retryIntervalMs = 2 * 60 * 1000;
        if (this.speedtestInstallAttempted && (now - this.speedtestLastInstallAttemptMs) < retryIntervalMs) {
            return;
        }
        this.speedtestInstallAttempted = true;
        this.speedtestLastInstallAttemptMs = now;
        const installAttempts = this.getSpeedtestInstallAttempts();
        for (const installCmd of installAttempts) {
            try {
                await this.runShellCommand(installCmd, 120000);
                for (const check of checks) {
                    try {
                        await this.runShellCommand(check, 10000);
                        return;
                    }
                    catch {
                        // continue checks
                    }
                }
            }
            catch {
                // continue installers
            }
        }
    }
    async handleSpeedtestRun(payload) {
        const requestId = String(payload?.requestId || '').trim();
        const agentId = this.wsClient.getAgentId() || config_1.config.agent.hostname;
        await this.ensureSpeedtestCliInstalled();
        const runtimeNode = String(process.execPath || '').replace(/"/g, '\\"');
        const localSpeedtestCmd = path_1.default.resolve(__dirname, '../node_modules/.bin/speedtest-cli.cmd').replace(/\//g, '\\');
        const localSpeedtestCliJs = path_1.default.resolve(__dirname, '../node_modules/speedtest-cli/bin/cli.js').replace(/\//g, '\\');
        const attempts = process.platform === 'win32'
            ? [
                `cmd /c if exist "${localSpeedtestCmd}" ("${localSpeedtestCmd}" --json --secure --share) else exit /b 1`,
                runtimeNode
                    ? `cmd /c if exist "${localSpeedtestCliJs}" ("${runtimeNode}" "${localSpeedtestCliJs}" --json --secure --share) else exit /b 1`
                    : '',
                'cmd /c speedtest --accept-license --accept-gdpr --format=json',
                'cmd /c speedtest --accept-license --accept-gdpr -f json',
                'cmd /c if exist "%ProgramFiles%\\Ookla\\Speedtest\\speedtest.exe" ("%ProgramFiles%\\Ookla\\Speedtest\\speedtest.exe" --accept-license --accept-gdpr --format=json) else exit /b 1',
                'cmd /c if exist "%ProgramFiles%\\Ookla\\Speedtest\\speedtest.exe" ("%ProgramFiles%\\Ookla\\Speedtest\\speedtest.exe" --accept-license --accept-gdpr -f json) else exit /b 1',
                'cmd /c if exist "%ProgramFiles(x86)%\\Ookla\\Speedtest\\speedtest.exe" ("%ProgramFiles(x86)%\\Ookla\\Speedtest\\speedtest.exe" --accept-license --accept-gdpr --format=json) else exit /b 1',
                'cmd /c if exist "%ProgramFiles(x86)%\\Ookla\\Speedtest\\speedtest.exe" ("%ProgramFiles(x86)%\\Ookla\\Speedtest\\speedtest.exe" --accept-license --accept-gdpr -f json) else exit /b 1',
                'cmd /c npx --yes speedtest-cli --json --secure --share',
                'cmd /c if exist "%AppData%\\npm\\speedtest-cli.cmd" ("%AppData%\\npm\\speedtest-cli.cmd" --json --secure --share) else exit /b 1',
                'cmd /c if exist "%AppData%\\npm\\speedtest.cmd" ("%AppData%\\npm\\speedtest.cmd" --json --secure --share) else exit /b 1',
                'cmd /c if exist "%ProgramData%\\AppStats\\npm\\speedtest-cli.cmd" ("%ProgramData%\\AppStats\\npm\\speedtest-cli.cmd" --json --secure --share) else exit /b 1',
                'cmd /c if exist "%ProgramData%\\AppStats\\npm\\speedtest.cmd" ("%ProgramData%\\AppStats\\npm\\speedtest.cmd" --json --secure --share) else exit /b 1',
                'cmd /c if exist "%ProgramFiles%\\nodejs\\node.exe" if exist "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" ("%ProgramFiles%\\nodejs\\node.exe" "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" --json --secure --share) else exit /b 1',
                'cmd /c if exist "%ProgramFiles(x86)%\\nodejs\\node.exe" if exist "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" ("%ProgramFiles(x86)%\\nodejs\\node.exe" "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" --json --secure --share) else exit /b 1',
                runtimeNode
                    ? `cmd /c if exist "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" ("${runtimeNode}" "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" --json --secure --share) else exit /b 1`
                    : '',
                'cmd /c if exist "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" (node "%ProgramData%\\AppStats\\npm\\node_modules\\speedtest-cli\\bin\\cli.js" --json --secure --share) else exit /b 1',
                'cmd /c speedtest-cli --json --secure --share',
                'cmd /c speedtest-cli --secure --share',
                'cmd /c py -3 -m speedtest --json --secure --share',
                'cmd /c py -m speedtest --json --secure --share',
                'cmd /c python -m speedtest --json --secure --share',
                'cmd /c python3 -m speedtest --json --secure --share',
                `powershell -NoProfile -Command "$exe=(Get-ChildItem -Path $env:APPDATA\\Python -Recurse -Filter speedtest-cli.exe -ErrorAction SilentlyContinue | Select-Object -First 1 -ExpandProperty FullName); if($exe){ & $exe --json --secure --share }"`,
                'cmd /c speedtest --accept-license --accept-gdpr',
            ].filter((cmd) => !!String(cmd).trim())
            : [
                'speedtest-cli --json --secure --share',
                'speedtest --accept-license --accept-gdpr --format=json',
                'speedtest --accept-license --accept-gdpr -f json',
                'python -m speedtest --json --secure --share',
                'python3 -m speedtest --json --secure --share',
                'speedtest --accept-license --accept-gdpr',
            ];
        let parsed = null;
        let textSummary = null;
        let usedCommand = '';
        let lastError = '';
        let rawResult = null;
        const commandTimeoutMs = process.platform === 'win32' ? 240000 : 180000;
        const executeAttempts = async (cmds) => {
            for (const cmd of cmds) {
                try {
                    const raw = await this.runShellCommand(cmd, commandTimeoutMs);
                    const data = this.parseJsonOutput(raw);
                    if (data && typeof data === 'object') {
                        parsed = data;
                        usedCommand = cmd;
                        rawResult = data;
                        return;
                    }
                    const parsedText = this.parseSpeedtestTextOutput(raw);
                    if (parsedText) {
                        textSummary = parsedText;
                        usedCommand = cmd;
                        rawResult = { output: raw };
                        return;
                    }
                    lastError = 'Speedtest output was not valid JSON';
                }
                catch (error) {
                    lastError = error?.message || String(error);
                }
            }
        };
        await executeAttempts(attempts);
        if (!parsed && process.platform === 'win32' && /not recognized as an internal or external command/i.test(String(lastError || ''))) {
            const installAttempts = this.getSpeedtestInstallAttempts();
            for (const installCmd of installAttempts) {
                try {
                    await this.runShellCommand(installCmd, 120000);
                    break;
                }
                catch {
                    // continue trying other installers
                }
            }
            await executeAttempts(attempts);
        }
        if (!parsed && !textSummary) {
            const missingCli = /(not recognized as an internal or external command|command not found|cannot find|no such file)/i.test(String(lastError || ''));
            this.wsClient.send({
                type: 'speedtest:result',
                payload: {
                    requestId,
                    agentId,
                    success: false,
                    error: missingCli
                        ? 'No speedtest CLI found on agent. Install speedtest-cli (CMD) or Ookla speedtest CLI.'
                        : (lastError || 'No speedtest command available on agent'),
                    timestamp: new Date().toISOString(),
                },
            });
            return;
        }
        let downloadMbps = 0;
        let uploadMbps = 0;
        let pingMs = 0;
        let jitterMs = null;
        let serverName = '';
        let serverLocation = '';
        let sponsor = '';
        let imageUrl = null;
        if (parsed) {
            const isOokla = parsed.download && typeof parsed.download === 'object' && parsed.download.bandwidth != null;
            downloadMbps = isOokla
                ? (Number(parsed.download?.bandwidth || 0) * 8) / 1000000
                : Number(parsed.download || 0) / 1000000;
            uploadMbps = isOokla
                ? (Number(parsed.upload?.bandwidth || 0) * 8) / 1000000
                : Number(parsed.upload || 0) / 1000000;
            pingMs = isOokla ? Number(parsed.ping?.latency || 0) : Number(parsed.ping || 0);
            jitterMs = isOokla ? Number(parsed.ping?.jitter || 0) : null;
            serverName = String(parsed.server?.name || parsed.server?.host || '');
            const locationBits = [parsed.server?.location, parsed.server?.country].filter((v) => !!v).map((v) => String(v));
            serverLocation = locationBits.join(', ');
            sponsor = String(parsed.server?.sponsor || '');
            imageUrl = String(parsed.share || parsed.result?.url || '').trim() || null;
        }
        else if (textSummary) {
            downloadMbps = Number(textSummary.downloadMbps || 0);
            uploadMbps = Number(textSummary.uploadMbps || 0);
            pingMs = Number(textSummary.pingMs || 0);
            jitterMs = Number.isFinite(Number(textSummary.jitterMs)) ? Number(textSummary.jitterMs) : null;
            serverName = String(textSummary.serverName || '');
            serverLocation = String(textSummary.serverLocation || '');
            sponsor = String(textSummary.sponsor || '');
            imageUrl = textSummary.imageUrl ? String(textSummary.imageUrl) : null;
        }
        this.wsClient.send({
            type: 'speedtest:result',
            payload: {
                requestId,
                agentId,
                success: true,
                command: usedCommand,
                summary: {
                    downloadMbps: Number.isFinite(downloadMbps) ? Math.round(downloadMbps * 100) / 100 : 0,
                    uploadMbps: Number.isFinite(uploadMbps) ? Math.round(uploadMbps * 100) / 100 : 0,
                    pingMs: Number.isFinite(pingMs) ? Math.round(pingMs * 100) / 100 : 0,
                    jitterMs: Number.isFinite(Number(jitterMs)) ? Math.round(Number(jitterMs) * 100) / 100 : null,
                    serverName,
                    serverLocation,
                    sponsor,
                },
                imageUrl,
                raw: rawResult || parsed || textSummary,
                timestamp: new Date().toISOString(),
            },
        });
    }
    flushProbeResults() {
        if (!this.wsClient.isConnected()) {
            return;
        }
        let hasPending = false;
        for (const [bufKey, buf] of this.probeResultBuffers.entries()) {
            if (buf.results.length === 0)
                continue;
            hasPending = true;
            this.wsClient.send({
                type: buf.type,
                payload: { probeId: buf.probeId, results: [...buf.results] },
                agentId: this.wsClient.getAgentId() || undefined,
            });
            logger_1.logger.info(`Flushed ${buf.results.length} ${buf.type} results for probe ${buf.probeId}`);
            buf.results = [];
        }
        if (!hasPending && this.probeTimers.size === 0 && this.probeFlushTimer) {
            clearInterval(this.probeFlushTimer);
            this.probeFlushTimer = null;
        }
    }
    // ---- Security Scan Handler ----
    async handleSecurityScan(payload) {
        const { scanId, scanType, target, portRange, options } = payload || {};
        const agentId = this.wsClient.getAgentId() || config_1.config.agent.hostname;
        const normalizedScanType = String(scanType || 'port').toLowerCase();
        const scanTarget = normalizedScanType === 'vlan' ? (target || 'local') : (target || 'localhost');
        logger_1.logger.info(`Security scan requested: ${normalizedScanType} on ${scanTarget} (scanId: ${scanId})`);
        try {
            const result = await this.securityScanner.scan(scanId, normalizedScanType, scanTarget, portRange || '1-1024', options || {}, (progress) => {
                // Send progress updates
                this.wsClient.sendMetrics('security:scan:progress', {
                    scanId,
                    progress: Math.round(progress),
                });
            });
            result.agentId = agentId;
            // Send results back to server
            this.wsClient.send({
                type: 'security:scan:complete',
                payload: {
                    scanId,
                    agentId,
                    status: result.error ? 'error' : 'completed',
                    results: result,
                    summary: result.summary,
                    completedAt: new Date().toISOString(),
                },
            });
            logger_1.logger.info(`Security scan complete: ${scanId} - ${result.summary.openPorts} open ports, ${result.findings.length} findings`);
        }
        catch (error) {
            logger_1.logger.error(`Security scan failed: ${error.message}`);
            this.wsClient.send({
                type: 'security:scan:complete',
                payload: {
                    scanId,
                    agentId,
                    status: 'error',
                    error: error.message,
                    completedAt: new Date().toISOString(),
                },
            });
        }
    }
    /**
     * Check whether a Windows service is installed, its start type, and running state.
     * Returns null on non-Windows or if child_process fails.
     */
    async getWindowsServiceState(serviceName) {
        if (process.platform !== 'win32')
            return null;
        try {
            const { execSync } = require('child_process');
            const raw = execSync(`powershell -NoProfile -NonInteractive -Command "Get-Service -Name '${serviceName}' -ErrorAction SilentlyContinue | Select-Object Status,StartType | ConvertTo-Json"`, { timeout: 8000 }).toString().trim();
            if (!raw)
                return { exists: false, running: false, startType: 'Unknown' };
            const obj = JSON.parse(raw);
            return {
                exists: true,
                running: typeof obj.Status === 'string'
                    ? obj.Status === 'Running'
                    : obj.Status === 4, // ServiceControllerStatus.Running = 4
                startType: typeof obj.StartType === 'string' ? obj.StartType : String(obj.StartType),
            };
        }
        catch {
            return null;
        }
    }
    /**
     * TCP probe — returns true if the port is already accepting connections.
     */
    tcpProbe(host, port, timeoutMs = 5000) {
        return new Promise((resolve) => {
            const socket = new net_1.default.Socket();
            socket.setTimeout(timeoutMs);
            socket.on('connect', () => { socket.destroy(); resolve(true); });
            socket.on('timeout', () => { socket.destroy(); resolve(false); });
            socket.on('error', () => { socket.destroy(); resolve(false); });
            socket.connect(port, host);
        });
    }
    /**
     * Ensure SSH (OpenSSH) is available and listening on the local machine.
     * Only called when the TCP probe says the port is NOT yet open.
     *
     * Returns a cleanup descriptor so we know what to undo later.
     */
    async ensureSSHAvailable(wgSourceIp) {
        const { execSync } = require('child_process');
        const cleanup = {
            weStartedService: false,
            weEnabledService: false,
            weAddedFirewallRule: false,
            firewallRuleName: `AppStats-SSH-Temp-${Date.now()}`,
            originalStartType: undefined,
        };
        const svcState = await this.getWindowsServiceState('sshd');
        if (!svcState || !svcState.exists) {
            // Try to install OpenSSH Server (Windows 10/11 Optional Feature)
            logger_1.logger.info('SSH: sshd not installed — attempting to add OpenSSH.Server capability');
            execSync('powershell -NoProfile -NonInteractive -Command "Add-WindowsCapability -Online -Name OpenSSH.Server~~~~0.0.1.0"', { timeout: 120000 });
        }
        // Re-check after potential install
        const svcState2 = await this.getWindowsServiceState('sshd');
        if (!svcState2 || !svcState2.exists) {
            throw new Error('OpenSSH Server is not available on this machine and could not be installed automatically.');
        }
        cleanup.originalStartType = svcState2.startType;
        if (svcState2.startType === 'Disabled') {
            logger_1.logger.info('SSH: service is Disabled — temporarily enabling (Manual start)');
            execSync('sc.exe config sshd start= demand', { timeout: 8000 });
            cleanup.weEnabledService = true;
        }
        if (!svcState2.running) {
            logger_1.logger.info('SSH: starting sshd service');
            execSync('net start sshd', { timeout: 15000 });
            cleanup.weStartedService = true;
        }
        // Add a scoped inbound firewall rule: only allow traffic from the WireGuard IP
        const ruleName = cleanup.firewallRuleName;
        logger_1.logger.info(`SSH: adding scoped firewall rule '${ruleName}' (allow from ${wgSourceIp})`);
        execSync(`netsh advfirewall firewall add rule name="${ruleName}" protocol=TCP dir=in localport=22 action=allow remoteip="${wgSourceIp}"`, { timeout: 8000 });
        cleanup.weAddedFirewallRule = true;
        // Wait briefly for the service to start listening
        await new Promise(r => setTimeout(r, 2000));
        return cleanup;
    }
    /**
     * Ensure RDP (Remote Desktop) is available on the local machine.
     * Only called when the TCP probe says port 3389 is NOT yet open.
     */
    async ensureRDPAvailable(wgSourceIp) {
        const { execSync } = require('child_process');
        const cleanup = {
            weStartedService: false,
            weEnabledService: false,
            weAddedFirewallRule: false,
            firewallRuleName: `AppStats-RDP-Temp-${Date.now()}`,
            originalStartType: undefined,
        };
        const svcState = await this.getWindowsServiceState('TermService');
        if (!svcState || !svcState.exists) {
            throw new Error('Remote Desktop Services (TermService) is not installed on this machine.');
        }
        cleanup.originalStartType = svcState.startType;
        if (svcState.startType === 'Disabled') {
            logger_1.logger.info('RDP: TermService is Disabled — temporarily enabling (Manual start)');
            execSync('sc.exe config TermService start= demand', { timeout: 8000 });
            execSync('sc.exe config UmRdpService start= demand', { timeout: 8000 });
            cleanup.weEnabledService = true;
        }
        if (!svcState.running) {
            logger_1.logger.info('RDP: starting TermService');
            execSync('net start TermService', { timeout: 15000 });
            cleanup.weStartedService = true;
        }
        // Enable RDP via registry if it's blocked there
        execSync('reg add "HKLM\\SYSTEM\\CurrentControlSet\\Control\\Terminal Server" /v fDenyTSConnections /t REG_DWORD /d 0 /f', { timeout: 5000 });
        // Add scoped inbound firewall rule: only allow traffic from the WireGuard source IP
        const ruleName = cleanup.firewallRuleName;
        logger_1.logger.info(`RDP: adding scoped firewall rule '${ruleName}' (allow from ${wgSourceIp})`);
        execSync(`netsh advfirewall firewall add rule name="${ruleName}" protocol=TCP dir=in localport=3389 action=allow remoteip="${wgSourceIp}"`, { timeout: 8000 });
        cleanup.weAddedFirewallRule = true;
        await new Promise(r => setTimeout(r, 2000));
        return cleanup;
    }
    /**
     * Tear down everything we set up for a session that was NOT previously running.
     */
    async tearDownRemoteService(sessionId) {
        const state = this.remoteServiceState.get(sessionId);
        if (!state)
            return;
        this.remoteServiceState.delete(sessionId);
        const { execSync } = require('child_process');
        // Remove the scoped firewall rule we added
        if (state.weAddedFirewallRule && state.firewallRuleName) {
            try {
                logger_1.logger.info(`Remote teardown: removing firewall rule '${state.firewallRuleName}'`);
                execSync(`netsh advfirewall firewall delete rule name="${state.firewallRuleName}"`, { timeout: 8000 });
            }
            catch (e) {
                logger_1.logger.warn(`Remote teardown: failed to remove firewall rule — ${e.message}`);
            }
        }
        const svcName = state.type === 'ssh' ? 'sshd' : 'TermService';
        // Stop the service only if WE started it
        if (state.weStartedService) {
            try {
                logger_1.logger.info(`Remote teardown: stopping ${svcName} (we started it)`);
                execSync(`net stop ${svcName} /y`, { timeout: 15000 });
            }
            catch (e) {
                logger_1.logger.warn(`Remote teardown: could not stop ${svcName} — ${e.message}`);
            }
        }
        // Restore the original start type only if WE changed it
        if (state.weEnabledService && state.originalStartType) {
            try {
                // Map PowerShell StartType back to sc.exe token
                const scToken = state.originalStartType === 'Disabled' ? 'disabled' : 'demand';
                logger_1.logger.info(`Remote teardown: restoring ${svcName} start type to '${scToken}'`);
                execSync(`sc.exe config ${svcName} start= ${scToken}`, { timeout: 8000 });
                if (state.type === 'rdp') {
                    // Also revert fDenyTSConnections and UmRdpService
                    execSync('reg add "HKLM\\SYSTEM\\CurrentControlSet\\Control\\Terminal Server" /v fDenyTSConnections /t REG_DWORD /d 1 /f', { timeout: 5000 });
                    execSync(`sc.exe config UmRdpService start= ${scToken}`, { timeout: 8000 });
                }
            }
            catch (e) {
                logger_1.logger.warn(`Remote teardown: failed to restore service start type — ${e.message}`);
            }
        }
        logger_1.logger.info(`Remote teardown complete for session ${sessionId} (type: ${state.type})`);
    }
    async handleRemoteSessionStart(payload) {
        const { sessionId, sessionType, targetHost, serviceHost, targetPort, wgSourceIp } = payload || {};
        const agentId = this.wsClient.getAgentId() || config_1.config.agent.hostname;
        // targetHost is the externally reachable host (prefer WireGuard IP).
        // serviceHost is what the agent should probe/manage locally.
        const externalHost = targetHost || 'localhost';
        const host = serviceHost || targetHost || 'localhost';
        const isSSH = (sessionType || '').toLowerCase() === 'ssh';
        const isRDP = (sessionType || '').toLowerCase() === 'rdp';
        const defaultPort = isSSH ? 22 : isRDP ? 3389 : (targetPort || 22);
        const port = targetPort || defaultPort;
        // The WireGuard peer IP to scope firewall rules to (fall back to any if not supplied)
        const sourceIp = wgSourceIp || '10.0.0.0/8';
        logger_1.logger.info(`Remote session start: ${sessionType} -> service ${host}:${port} (access ${externalHost}:${port}) (session: ${sessionId})`);
        try {
            // ── Step 1: probe whether the service is already listening ──────────────
            const alreadyListening = await this.tcpProbe(host, port, 3000);
            if (alreadyListening) {
                // Service is already up and running — do NOT touch anything.
                logger_1.logger.info(`Remote session ${sessionId}: ${sessionType.toUpperCase()} already listening on ${host}:${port} — no service changes made`);
                this.activeSessions.set(sessionId, {
                    sessionId, sessionType, targetHost: externalHost, targetPort: port,
                    managedByUs: false,
                    startedAt: new Date().toISOString(),
                });
            }
            else {
                // ── Step 2: service not listening — enable it securely ────────────────
                if (process.platform !== 'win32') {
                    // Linux: check and enable via systemctl
                    const { execSync } = require('child_process');
                    const svcName = isRDP ? 'xrdp' : 'ssh';
                    try {
                        const status = execSync(`systemctl is-active ${svcName} 2>/dev/null`, { timeout: 5000 }).toString().trim();
                        if (status !== 'active') {
                            logger_1.logger.info(`Remote session ${sessionId}: starting ${svcName} on Linux`);
                            execSync(`systemctl start ${svcName}`, { timeout: 15000 });
                            this.remoteServiceState.set(sessionId, {
                                type: isRDP ? 'rdp' : 'ssh',
                                weStartedService: true,
                                weEnabledService: false,
                                weAddedFirewallRule: false,
                                firewallRuleName: '',
                            });
                        }
                    }
                    catch {
                        throw new Error(`Could not start ${svcName} on Linux — please ensure it is installed.`);
                    }
                    await new Promise(r => setTimeout(r, 2000));
                }
                else {
                    // Windows: full service + firewall management
                    logger_1.logger.info(`Remote session ${sessionId}: ${sessionType.toUpperCase()} not listening — enabling securely`);
                    let svcCleanup;
                    if (isSSH) {
                        svcCleanup = await this.ensureSSHAvailable(sourceIp);
                    }
                    else if (isRDP) {
                        svcCleanup = await this.ensureRDPAvailable(sourceIp);
                    }
                    else {
                        throw new Error(`Unsupported session type '${sessionType}' — cannot auto-enable service`);
                    }
                    this.remoteServiceState.set(sessionId, { type: isSSH ? 'ssh' : 'rdp', ...svcCleanup });
                }
                // Re-probe to confirm the service is now up
                const nowListening = await this.tcpProbe(host, port, 8000);
                if (!nowListening) {
                    // Clean up what we did, then report failure
                    await this.tearDownRemoteService(sessionId);
                    throw new Error(`${sessionType.toUpperCase()} service was started but ${host}:${port} is still not reachable`);
                }
                logger_1.logger.info(`Remote session ${sessionId}: service enabled and confirmed listening on ${host}:${port}`);
                this.activeSessions.set(sessionId, {
                    sessionId, sessionType, targetHost: externalHost, targetPort: port,
                    managedByUs: true,
                    startedAt: new Date().toISOString(),
                });
            }
            // ── Step 3: Report session active ────────────────────────────────────────
            const sessionState = this.activeSessions.get(sessionId);
            this.wsClient.send({
                type: 'remote:session:data',
                payload: {
                    sessionId,
                    agentId,
                    status: 'active',
                    message: `${sessionType.toUpperCase()} session ready on ${externalHost}:${port}`,
                    managedByUs: sessionState.managedByUs,
                    connectionDetails: { host: externalHost, serviceHost: host, port, type: sessionType },
                },
            });
            logger_1.logger.info(`Remote session ${sessionId} active`);
        }
        catch (error) {
            logger_1.logger.error(`Remote session error: ${error.message}`);
            this.wsClient.send({
                type: 'remote:session:data',
                payload: {
                    sessionId,
                    agentId,
                    status: 'error',
                    error: error.message,
                },
            });
        }
    }
    async handleRemoteSessionEnd(payload) {
        const { sessionId } = payload || {};
        const agentId = this.wsClient.getAgentId() || config_1.config.agent.hostname;
        const sessionInfo = this.activeSessions.get(sessionId);
        logger_1.logger.info(`Remote session end: ${sessionId} (managedByUs: ${sessionInfo?.managedByUs ?? false})`);
        this.activeSessions.delete(sessionId);
        // If we enabled the service for this session, tear it down
        if (sessionInfo?.managedByUs) {
            await this.tearDownRemoteService(sessionId);
        }
        this.wsClient.send({
            type: 'remote:session:end',
            payload: {
                sessionId,
                agentId,
                status: 'ended',
                endedAt: new Date().toISOString(),
            },
        });
    }
    async installBrowserExtension(payload) {
        const { serverUrl } = payload || {};
        const requestedBrowser = String(payload?.browser || 'edge').toLowerCase();
        const browserAliases = {
            msedge: 'edge',
            chromium: 'chrome',
            opera_gx: 'operagx',
            gx: 'operagx',
        };
        const normalizedBrowser = browserAliases[requestedBrowser] || requestedBrowser;
        const supportedBrowsers = ['edge', 'chrome', 'brave', 'opera', 'operagx', 'firefox'];
        const targetBrowsers = normalizedBrowser === 'all'
            ? supportedBrowsers
            : (supportedBrowsers.includes(normalizedBrowser) ? [normalizedBrowser] : ['edge']);
        const agentId = this.wsClient.getAgentId() || config_1.config.agent.hostname;
        logger_1.logger.info(`Browser extension install requested`, { requestedBrowser, targetBrowsers, serverUrl });
        try {
            // 1. Download the extension zip from the server
            const baseUrl = (serverUrl || config_1.config.server.url || this.wgDeployUrl || 'http://localhost').replace(/\/+$/, '');
            const downloadCandidates = Array.from(new Set([
                baseUrl,
                (config_1.config.server.url || '').replace(/\/+$/, ''),
                (this.wgDeployUrl || '').replace(/\/+$/, ''),
            ].filter(Boolean))).map((u) => `${u}/api/upgrades/packages/appstats-browser-extension.zip`);
            const dataDir = path_1.default.join(process.env.PROGRAMDATA || 'C:\\ProgramData', 'AppStats');
            const extDir = path_1.default.join(dataDir, 'browser-extension');
            const zipPath = path_1.default.join(extDir, 'appstats-browser-extension.zip');
            // Create directory
            fs_1.default.mkdirSync(extDir, { recursive: true });
            const downloadToFile = (url, allowInsecureTls = false, redirectDepth = 0) => new Promise((resolve, reject) => {
                if (redirectDepth > 5) {
                    reject(new Error('Too many redirects'));
                    return;
                }
                const urlObj = new url_1.URL(url);
                const client = urlObj.protocol === 'https:' ? https_1.default : http_1.default;
                const options = {};
                if (urlObj.protocol === 'https:' && allowInsecureTls) {
                    options.rejectUnauthorized = false;
                }
                const req = client.get(url, options, (res) => {
                    const status = Number(res.statusCode || 0);
                    if ((status === 301 || status === 302 || status === 307 || status === 308) && res.headers.location) {
                        const nextUrl = new url_1.URL(res.headers.location, url).toString();
                        res.resume();
                        downloadToFile(nextUrl, allowInsecureTls, redirectDepth + 1).then(resolve).catch(reject);
                        return;
                    }
                    if (status !== 200) {
                        res.resume();
                        reject(new Error(`Download failed: HTTP ${status}`));
                        return;
                    }
                    const file = fs_1.default.createWriteStream(zipPath);
                    file.on('error', reject);
                    res.on('error', reject);
                    file.on('finish', () => {
                        file.close();
                        resolve();
                    });
                    res.pipe(file);
                });
                req.on('error', reject);
            });
            let downloadedFrom = '';
            const errors = [];
            for (const downloadUrl of downloadCandidates) {
                try {
                    logger_1.logger.info(`Downloading extension from ${downloadUrl}`);
                    await downloadToFile(downloadUrl, false);
                    downloadedFrom = downloadUrl;
                    break;
                }
                catch (e1) {
                    if (downloadUrl.startsWith('https://')) {
                        try {
                            logger_1.logger.warn(`Extension download failed over strict TLS, retrying insecure TLS: ${e1.message}`);
                            await downloadToFile(downloadUrl, true);
                            downloadedFrom = downloadUrl;
                            break;
                        }
                        catch (e2) {
                            errors.push(`${downloadUrl} (strict: ${e1.message}; insecure: ${e2.message})`);
                        }
                    }
                    else {
                        errors.push(`${downloadUrl} (${e1.message})`);
                    }
                }
            }
            if (!downloadedFrom) {
                throw new Error(`Extension download failed from all candidates: ${errors.join(' | ')}`);
            }
            logger_1.logger.info(`Extension downloaded from ${downloadedFrom}, extracting...`);
            // 2. Extract zip using PowerShell (Windows)
            const extractDir = path_1.default.join(extDir, 'unpacked');
            if (fs_1.default.existsSync(extractDir)) {
                fs_1.default.rmSync(extractDir, { recursive: true, force: true });
            }
            fs_1.default.mkdirSync(extractDir, { recursive: true });
            const { execSync } = require('child_process');
            execSync(`powershell -Command "Expand-Archive -Path '${zipPath}' -DestinationPath '${extractDir}' -Force"`, { timeout: 30000 });
            logger_1.logger.info('Extension extracted to ' + extractDir);
            // 3. Configure extension — write/update config in the extension
            const configPath = path_1.default.join(extractDir, 'config.json');
            const extConfig = {
                serverUrl: baseUrl.replace(/\/+$/, ''),
                agentId: agentId,
                autoConnect: true
            };
            fs_1.default.writeFileSync(configPath, JSON.stringify(extConfig, null, 2));
            // 4. Build extension artifacts for force-install/update.
            let extensionVersion = '1.0.0';
            try {
                const manifestJson = JSON.parse(fs_1.default.readFileSync(path_1.default.join(extractDir, 'manifest.json'), 'utf8'));
                extensionVersion = String(manifestJson?.version || extensionVersion);
            }
            catch {
                // keep default
            }
            const chromiumTargets = targetBrowsers.filter((b) => b !== 'firefox');
            const chromiumPackerCandidates = [
                path_1.default.join(process.env['ProgramFiles(x86)'] || '', 'Microsoft', 'Edge', 'Application', 'msedge.exe'),
                path_1.default.join(process.env.PROGRAMFILES || '', 'Microsoft', 'Edge', 'Application', 'msedge.exe'),
                path_1.default.join(process.env.PROGRAMFILES || '', 'Google', 'Chrome', 'Application', 'chrome.exe'),
                path_1.default.join(process.env['ProgramFiles(x86)'] || '', 'Google', 'Chrome', 'Application', 'chrome.exe'),
                path_1.default.join(process.env.LOCALAPPDATA || '', 'BraveSoftware', 'Brave-Browser', 'Application', 'brave.exe'),
                path_1.default.join(process.env.LOCALAPPDATA || '', 'Programs', 'Opera', 'launcher.exe'),
                path_1.default.join(process.env.LOCALAPPDATA || '', 'Programs', 'Opera GX', 'launcher.exe'),
            ];
            const chromiumPackerExe = chromiumPackerCandidates.find((p) => p && fs_1.default.existsSync(p));
            const extensionPemPath = path_1.default.join(extDir, 'appstats-browser-extension.pem');
            const generatedPemPath = `${extractDir}.pem`;
            const generatedCrxPath = `${extractDir}.crx`;
            const packagedCrxPath = path_1.default.join(extDir, 'appstats-browser-extension.crx');
            const packagedXpiPath = path_1.default.join(extDir, 'appstats-browser-extension.xpi');
            const updateXmlPath = path_1.default.join(extDir, 'updates.xml');
            let chromiumExtensionId = '';
            let chromiumUpdateUrl = '';
            try {
                fs_1.default.copyFileSync(zipPath, packagedXpiPath);
            }
            catch { }
            if (chromiumTargets.length > 0) {
                if (chromiumPackerExe) {
                    try {
                        if (fs_1.default.existsSync(generatedCrxPath))
                            fs_1.default.rmSync(generatedCrxPath, { force: true });
                        if (fs_1.default.existsSync(generatedPemPath))
                            fs_1.default.rmSync(generatedPemPath, { force: true });
                        const packCmd = fs_1.default.existsSync(extensionPemPath)
                            ? `"${chromiumPackerExe}" --pack-extension="${extractDir}" --pack-extension-key="${extensionPemPath}"`
                            : `"${chromiumPackerExe}" --pack-extension="${extractDir}"`;
                        execSync(packCmd, { timeout: 120000, windowsHide: true });
                        if (!fs_1.default.existsSync(extensionPemPath) && fs_1.default.existsSync(generatedPemPath)) {
                            fs_1.default.copyFileSync(generatedPemPath, extensionPemPath);
                        }
                        if (fs_1.default.existsSync(generatedCrxPath)) {
                            fs_1.default.copyFileSync(generatedCrxPath, packagedCrxPath);
                        }
                        if (fs_1.default.existsSync(extensionPemPath) && fs_1.default.existsSync(packagedCrxPath)) {
                            chromiumExtensionId = this.computeChromiumExtensionIdFromPem(fs_1.default.readFileSync(extensionPemPath, 'utf8'));
                            if (chromiumExtensionId) {
                                chromiumUpdateUrl = 'http://127.0.0.1:9898/extensions/updates.xml';
                                const updateXml = `<?xml version='1.0' encoding='UTF-8'?>\n<gupdate xmlns='http://www.google.com/update2/response' protocol='2.0'>\n  <app appid='${chromiumExtensionId}'>\n    <updatecheck codebase='http://127.0.0.1:9898/extensions/appstats-browser-extension.crx' version='${extensionVersion}' />\n  </app>\n</gupdate>\n`;
                                fs_1.default.writeFileSync(updateXmlPath, updateXml);
                            }
                        }
                    }
                    catch (e) {
                        logger_1.logger.warn(`Chromium CRX packaging failed: ${e.message}`);
                    }
                }
                else {
                    logger_1.logger.warn('Chromium CRX packaging skipped: no browser executable found');
                }
            }
            // 5. Install/configure browser policies and native host for one or more browsers
            const browserPolicies = {
                chrome: {
                    devMode: 'HKLM\\SOFTWARE\\Policies\\Google\\Chrome',
                    forceInstall: 'HKLM\\SOFTWARE\\Policies\\Google\\Chrome\\ExtensionInstallForcelist',
                },
                edge: {
                    devMode: 'HKLM\\SOFTWARE\\Policies\\Microsoft\\Edge',
                    forceInstall: 'HKLM\\SOFTWARE\\Policies\\Microsoft\\Edge\\ExtensionInstallForcelist',
                },
                brave: {
                    devMode: 'HKLM\\SOFTWARE\\Policies\\BraveSoftware\\Brave',
                    forceInstall: 'HKLM\\SOFTWARE\\Policies\\BraveSoftware\\Brave\\ExtensionInstallForcelist',
                },
                opera: {
                    devMode: 'HKLM\\SOFTWARE\\Policies\\Opera Software\\Opera Stable',
                    forceInstall: 'HKLM\\SOFTWARE\\Policies\\Opera Software\\Opera Stable\\ExtensionInstallForcelist',
                },
                operagx: {
                    devMode: 'HKLM\\SOFTWARE\\Policies\\Opera Software\\Opera GX Stable',
                    forceInstall: 'HKLM\\SOFTWARE\\Policies\\Opera Software\\Opera GX Stable\\ExtensionInstallForcelist',
                },
                firefox: { devMode: 'HKLM\\SOFTWARE\\Policies\\Mozilla\\Firefox' },
            };
            // 6. Native messaging host (Firefox gets explicit extension ID allowlist)
            const nativeHostDirs = {
                chrome: path_1.default.join(process.env.LOCALAPPDATA || '', 'Google', 'Chrome', 'User Data', 'NativeMessagingHosts'),
                edge: path_1.default.join(process.env.LOCALAPPDATA || '', 'Microsoft', 'Edge', 'User Data', 'NativeMessagingHosts'),
                brave: path_1.default.join(process.env.LOCALAPPDATA || '', 'BraveSoftware', 'Brave-Browser', 'User Data', 'NativeMessagingHosts'),
                opera: path_1.default.join(process.env.APPDATA || '', 'Opera Software', 'Opera Stable', 'NativeMessagingHosts'),
                operagx: path_1.default.join(process.env.APPDATA || '', 'Opera Software', 'Opera GX Stable', 'NativeMessagingHosts'),
                firefox: path_1.default.join(process.env.APPDATA || '', 'Mozilla', 'NativeMessagingHosts'),
            };
            const nativeHostScript = path_1.default.join(extDir, 'native-host.bat');
            const nativeHostJs = path_1.default.join(extDir, 'native-host.js');
            const jsContent = `
const fs = require('fs');
const path = require('path');
const configPath = path.join(__dirname, 'unpacked', 'config.json');

// Read stdin length (first 4 bytes as uint32)
process.stdin.once('readable', () => {
  const lenBuf = process.stdin.read(4);
  if (!lenBuf) { process.exit(0); }
  const len = lenBuf.readUInt32LE(0);
  const msgBuf = process.stdin.read(len);
  if (!msgBuf) { process.exit(0); }

  try {
    const msg = JSON.parse(msgBuf.toString());
    let response = {};
    if (msg.type === 'get-config' && fs.existsSync(configPath)) {
      response = JSON.parse(fs.readFileSync(configPath, 'utf8'));
    }
    // Write response: 4-byte length prefix + JSON
    const out = Buffer.from(JSON.stringify(response));
    const header = Buffer.alloc(4);
    header.writeUInt32LE(out.length, 0);
    process.stdout.write(header);
    process.stdout.write(out);
  } catch (e) { /* ignore */ }
  process.exit(0);
});
`;
            fs_1.default.writeFileSync(nativeHostJs, jsContent.trim());
            fs_1.default.writeFileSync(nativeHostScript, `@echo off\r\nnode "${nativeHostJs}"`);
            const regKeys = {
                chrome: 'HKLM\\SOFTWARE\\Google\\Chrome\\NativeMessagingHosts\\com.appstats.agent',
                edge: 'HKLM\\SOFTWARE\\Microsoft\\Edge\\NativeMessagingHosts\\com.appstats.agent',
                brave: 'HKLM\\SOFTWARE\\BraveSoftware\\Brave\\NativeMessagingHosts\\com.appstats.agent',
                opera: 'HKLM\\SOFTWARE\\Opera Software\\Opera Stable\\NativeMessagingHosts\\com.appstats.agent',
                operagx: 'HKLM\\SOFTWARE\\Opera Software\\Opera GX Stable\\NativeMessagingHosts\\com.appstats.agent',
                firefox: 'HKLM\\SOFTWARE\\Mozilla\\NativeMessagingHosts\\com.appstats.agent',
            };
            const configuredBrowsers = [];
            const skippedBrowsers = [];
            for (const target of targetBrowsers) {
                const policy = browserPolicies[target];
                if (!policy) {
                    skippedBrowsers.push(target);
                    continue;
                }
                if (target !== 'firefox') {
                    try {
                        execSync(`reg add "${policy.devMode}" /v ExtensionDeveloperModeAllowed /t REG_DWORD /d 1 /f`, { timeout: 10000 });
                        logger_1.logger.info(`Enabled developer mode policy for ${target}`);
                    }
                    catch (e) {
                        logger_1.logger.warn(`Developer mode policy for ${target}: ${e.message}`);
                    }
                    if (policy.forceInstall && chromiumExtensionId && chromiumUpdateUrl && fs_1.default.existsSync(packagedCrxPath) && fs_1.default.existsSync(updateXmlPath)) {
                        try {
                            execSync(`reg add "${policy.forceInstall}" /v 1 /t REG_SZ /d "${chromiumExtensionId};${chromiumUpdateUrl}" /f`, { timeout: 10000 });
                        }
                        catch (e) {
                            logger_1.logger.warn(`Force-install policy for ${target}: ${e.message}`);
                        }
                        configuredBrowsers.push(target);
                    }
                    else {
                        skippedBrowsers.push(`${target} (missing CRX/update artifacts)`);
                    }
                    continue;
                }
                try {
                    const nativeHostDir = nativeHostDirs[target] || nativeHostDirs.edge;
                    fs_1.default.mkdirSync(nativeHostDir, { recursive: true });
                    const hostManifest = {
                        name: 'com.appstats.agent',
                        description: 'AppStats Agent Native Messaging Host',
                        path: nativeHostScript,
                        type: 'stdio',
                        allowed_extensions: ['appstats@emtelle.local'],
                    };
                    const manifestPath = path_1.default.join(nativeHostDir, 'com.appstats.agent.json');
                    fs_1.default.writeFileSync(manifestPath, JSON.stringify(hostManifest, null, 2));
                    execSync(`reg add "${regKeys[target] || regKeys.edge}" /ve /t REG_SZ /d "${manifestPath}" /f`, { timeout: 10000 });
                    if (fs_1.default.existsSync(packagedXpiPath)) {
                        const firefoxInstallUrl = `file:///${packagedXpiPath.replace(/\\/g, '/')}`;
                        const firefoxDistributions = [
                            path_1.default.join(process.env.PROGRAMFILES || '', 'Mozilla Firefox', 'distribution'),
                            path_1.default.join(process.env['ProgramFiles(x86)'] || '', 'Mozilla Firefox', 'distribution'),
                        ];
                        for (const distPath of firefoxDistributions) {
                            if (!distPath || !fs_1.default.existsSync(path_1.default.dirname(distPath)))
                                continue;
                            fs_1.default.mkdirSync(distPath, { recursive: true });
                            const policiesPath = path_1.default.join(distPath, 'policies.json');
                            let existing = {};
                            try {
                                existing = fs_1.default.existsSync(policiesPath) ? JSON.parse(fs_1.default.readFileSync(policiesPath, 'utf8')) : {};
                            }
                            catch {
                                existing = {};
                            }
                            existing.policies = existing.policies || {};
                            existing.policies.Extensions = existing.policies.Extensions || {};
                            const installList = Array.isArray(existing.policies.Extensions.Install) ? existing.policies.Extensions.Install : [];
                            if (!installList.includes(firefoxInstallUrl))
                                installList.push(firefoxInstallUrl);
                            existing.policies.Extensions.Install = installList;
                            fs_1.default.writeFileSync(policiesPath, JSON.stringify(existing, null, 2));
                        }
                    }
                    configuredBrowsers.push(target);
                }
                catch (e) {
                    skippedBrowsers.push(target);
                    logger_1.logger.warn(`Native messaging setup for ${target}: ${e.message}`);
                }
            }
            // Report result
            const extensionConfigured = configuredBrowsers.length > 0;
            this.wsClient.sendMetrics('extension:install:result', {
                success: extensionConfigured,
                browser: normalizedBrowser,
                extractDir,
                message: extensionConfigured
                    ? `Extension packaged and policy-configured for: ${configuredBrowsers.join(', ')}. Browser restart may be required to apply/refresh the extension.`
                    : `Extension package downloaded, but no target browser could be configured automatically.`,
                skippedBrowsers,
            });
            logger_1.logger.info(`Browser extension install complete`, { normalizedBrowser, configuredBrowsers, skippedBrowsers, extractDir });
        }
        catch (error) {
            logger_1.logger.error('Extension install failed:', error);
            this.wsClient.sendMetrics('extension:install:result', {
                success: false,
                browser: normalizedBrowser,
                error: error.message
            });
        }
    }
    // ---- WireGuard VPN Handler ----
    /**
     * Persist the original deploy URL into agent-config.json as `deployUrl` / `deployWsUrl`.
     * This is written ONCE on first run (never overwritten) so we always have the original
     * server IP/FQDN to fall back to if the WireGuard tunnel fails.
     */
    persistDeployUrl() {
        try {
            const configFile = path_1.default.join(path_1.default.dirname(process.execPath), 'agent-config.json');
            const existing = fs_1.default.existsSync(configFile) ? JSON.parse(fs_1.default.readFileSync(configFile, 'utf8')) : {};
            // Only write once — if already set, just read the saved value into memory
            if (existing.deployUrl && existing.deployWsUrl) {
                this.wgDeployUrl = existing.deployUrl;
                this.wgDeployWsUrl = existing.deployWsUrl;
                logger_1.logger.info(`WireGuard: deploy URL loaded from config → ${this.wgDeployUrl}`);
            }
            else {
                // First run: record the current URL as the deploy URL
                this.wgDeployUrl = config_1.config.server.url || 'http://localhost:3000';
                this.wgDeployWsUrl = config_1.config.server.wsUrl || 'ws://localhost:3001';
                fs_1.default.writeFileSync(configFile, JSON.stringify({
                    ...existing,
                    deployUrl: this.wgDeployUrl,
                    deployWsUrl: this.wgDeployWsUrl,
                }, null, 2));
                logger_1.logger.info(`WireGuard: deploy URL saved → ${this.wgDeployUrl}`);
            }
        }
        catch (e) {
            // Non-fatal — fall back to runtime config values
            this.wgDeployUrl = config_1.config.server.url || 'http://localhost:3000';
            this.wgDeployWsUrl = config_1.config.server.wsUrl || 'ws://localhost:3001';
            logger_1.logger.warn(`WireGuard: could not persist deploy URL: ${e.message}`);
        }
    }
    /**
     * Start a background health monitor that checks the WireGuard tunnel every 60 s.
     * If the tunnel goes stale (no handshake within 3 minutes) OR the WS connection
     * is lost and can't be restored via the WG URL, the agent falls back to the
     * original deploy URL automatically.
     *
     * The monitor also re-reports tunnel status (handshake age, rx/tx) to the server
     * on every tick so the dashboard stays current without agent restarts.
     */
    startWgHealthMonitor(agentId) {
        if (this.wgHealthMonitor) {
            clearInterval(this.wgHealthMonitor);
            this.wgHealthMonitor = null;
        }
        let consecutiveFails = 0;
        const MAX_FAILS = 3; // 3 × 60 s = 3 minutes before fallback
        const wgCmd = this.resolveWireGuardCommands().wg;
        this.wgHealthMonitor = setInterval(() => {
            const { exec } = require('child_process');
            exec(`${wgCmd} show`, { timeout: 5000 }, (err, stdout) => {
                try {
                    if (err) {
                        // wg show failed — tunnel may not be installed
                        consecutiveFails++;
                        if (consecutiveFails >= MAX_FAILS) {
                            clearInterval(this.wgHealthMonitor);
                            this.wgHealthMonitor = null;
                            this.wgActive = false;
                        }
                        return;
                    }
                    const wgOutput = stdout || '';
                    // Parse handshake timestamp
                    const handshakeMatch = wgOutput.match(/latest handshake: (.+)/);
                    const lastHandshake = handshakeMatch ? handshakeMatch[1].trim() : null;
                    // Parse traffic counters
                    const rxMatch = wgOutput.match(/transfer: ([\d.]+\s*\w+) received,\s*([\d.]+\s*\w+) sent/);
                    const parseBytes = (s) => {
                        const m = s.trim().match(/([\d.]+)\s*(\w+)/);
                        if (!m)
                            return 0;
                        const n = parseFloat(m[1]);
                        const u = m[2].toLowerCase();
                        if (u.startsWith('g'))
                            return Math.round(n * 1e9);
                        if (u.startsWith('m'))
                            return Math.round(n * 1e6);
                        if (u.startsWith('k'))
                            return Math.round(n * 1e3);
                        return Math.round(n);
                    };
                    const rxBytes = rxMatch ? parseBytes(rxMatch[1]) : 0;
                    const txBytes = rxMatch ? parseBytes(rxMatch[2]) : 0;
                    // Determine handshake age (WG reports e.g. "2 minutes, 35 seconds ago")
                    let handshakeAgeSeconds = Infinity;
                    if (lastHandshake && !lastHandshake.includes('(none)') && lastHandshake !== 'none') {
                        const minMatch = lastHandshake.match(/(\d+)\s+minute/);
                        const secMatch = lastHandshake.match(/(\d+)\s+second/);
                        const hrs = lastHandshake.match(/(\d+)\s+hour/);
                        handshakeAgeSeconds =
                            (hrs ? parseInt(hrs[1]) * 3600 : 0) +
                                (minMatch ? parseInt(minMatch[1]) * 60 : 0) +
                                (secMatch ? parseInt(secMatch[1]) : 0);
                    }
                    const tunnelAlive = handshakeAgeSeconds < 180; // < 3 min = healthy
                    // Report current status to server
                    this.wsClient.sendMetrics('wireguard:status', {
                        agentId, status: tunnelAlive ? 'up' : 'stale',
                        lastHandshake, rxBytes, txBytes,
                        handshakeAgeSeconds: handshakeAgeSeconds === Infinity ? null : handshakeAgeSeconds,
                    });
                    if (tunnelAlive) {
                        // Tunnel is healthy — reset failures even if WS is briefly disconnected
                        // (WS disconnects during reconnect are transient, not a WG problem)
                        consecutiveFails = 0;
                        return;
                    }
                    // Tunnel handshake is stale — count as failure
                    consecutiveFails++;
                    logger_1.logger.warn(`WireGuard health: tunnel stale (fail ${consecutiveFails}/${MAX_FAILS})`);
                    if (consecutiveFails >= MAX_FAILS) {
                        // Tunnel is dead — fall back to original deploy URL
                        consecutiveFails = 0;
                        this.wgActive = false;
                        clearInterval(this.wgHealthMonitor);
                        this.wgHealthMonitor = null;
                        logger_1.logger.warn(`WireGuard health: falling back to deploy URL → ${this.wgDeployUrl}`);
                        config_1.config.server.url = this.wgDeployUrl;
                        config_1.config.server.wsUrl = this.wgDeployWsUrl;
                        // Persist the revert
                        try {
                            const configFile = path_1.default.join(path_1.default.dirname(process.execPath), 'agent-config.json');
                            const existing = fs_1.default.existsSync(configFile) ? JSON.parse(fs_1.default.readFileSync(configFile, 'utf8')) : {};
                            fs_1.default.writeFileSync(configFile, JSON.stringify({
                                ...existing,
                                serverUrl: this.wgDeployUrl,
                                wsUrl: this.wgDeployWsUrl,
                            }, null, 2));
                        }
                        catch (_) { }
                        // Update browser extension config back to original URL
                        const extConfigPath = path_1.default.join(process.env.PROGRAMDATA || 'C:\\ProgramData', 'AppStats', 'browser-extension', 'unpacked', 'config.json');
                        if (fs_1.default.existsSync(extConfigPath)) {
                            try {
                                const extCfg = JSON.parse(fs_1.default.readFileSync(extConfigPath, 'utf8'));
                                extCfg.serverUrl = this.wgDeployUrl.replace(/\/+$/, '');
                                fs_1.default.writeFileSync(extConfigPath, JSON.stringify(extCfg, null, 2));
                            }
                            catch (_) { }
                        }
                        this.wsClient.reconnect();
                        this.wsClient.sendMetrics('wireguard:status', {
                            agentId, status: 'fallback',
                            message: `Tunnel stale — reverted to original server URL (${this.wgDeployUrl})`
                        });
                    }
                }
                catch (_) {
                    consecutiveFails++;
                    if (consecutiveFails >= MAX_FAILS) {
                        clearInterval(this.wgHealthMonitor);
                        this.wgHealthMonitor = null;
                        this.wgActive = false;
                    }
                }
            });
        }, 60000); // check every 60 seconds
    }
    getWgSplitTunnelAllowedIp(serverConfig, peerConfig) {
        const subnetRaw = String(serverConfig?.subnet || '').trim();
        const subnetIp = subnetRaw.split('/')[0];
        const isIpv4 = /^\d{1,3}(?:\.\d{1,3}){3}$/.test(subnetIp);
        const isFullTunnel = subnetRaw === '0.0.0.0/0' || subnetRaw === '::/0' || subnetIp === '0.0.0.0';
        if (isIpv4 && !isFullTunnel) {
            return `${subnetIp.replace(/\.\d+$/, '.1')}/32`;
        }
        const assignedIp = String(peerConfig?.assigned_ip || '').split('/')[0];
        if (/^\d{1,3}(?:\.\d{1,3}){3}$/.test(assignedIp)) {
            return `${assignedIp.replace(/\.\d+$/, '.1')}/32`;
        }
        return '10.100.0.1/32';
    }
    resolveWireGuardCommands() {
        if (process.platform !== 'win32') {
            return { wireguard: 'wireguard', wg: 'wg' };
        }
        const wireguardCandidates = [
            'C:\\Program Files\\WireGuard\\wireguard.exe',
            'C:\\Program Files (x86)\\WireGuard\\wireguard.exe',
        ];
        const wgCandidates = [
            'C:\\Program Files\\WireGuard\\wg.exe',
            'C:\\Program Files (x86)\\WireGuard\\wg.exe',
        ];
        const wireguardPath = wireguardCandidates.find((p) => fs_1.default.existsSync(p));
        const wgPath = wgCandidates.find((p) => fs_1.default.existsSync(p));
        return {
            wireguard: wireguardPath ? `"${wireguardPath}"` : 'wireguard',
            wg: wgPath ? `"${wgPath}"` : 'wg',
        };
    }
    computeChromiumExtensionIdFromPem(pem) {
        try {
            const crypto = require('crypto');
            const privateKey = crypto.createPrivateKey(pem);
            const publicDer = crypto.createPublicKey(privateKey).export({ type: 'spki', format: 'der' });
            const hash = crypto.createHash('sha256').update(publicDer).digest('hex').slice(0, 32);
            const alphabet = 'abcdefghijklmnop';
            return hash.split('').map((h) => alphabet[parseInt(h, 16)]).join('');
        }
        catch {
            return '';
        }
    }
    async handleWireGuardConfig(payload) {
        const { enabled, serverConfig, peerConfig } = payload || {};
        const agentId = this.wsClient.getAgentId() || config_1.config.agent.hostname;
        const wgTools = this.resolveWireGuardCommands();
        logger_1.logger.info(`WireGuard config received`, { enabled });
        try {
            const { execSync } = require('child_process');
            const wgConfPath = process.platform === 'win32'
                ? 'C:\\Program Files\\WireGuard\\Data\\Configurations\\wg0.conf'
                : '/etc/wireguard/wg0.conf';
            const wgDir = path_1.default.dirname(wgConfPath);
            if (!enabled || !serverConfig || !peerConfig) {
                // Disable / remove WireGuard
                logger_1.logger.info('WireGuard: disabling tunnel');
                // Stop health monitor
                if (this.wgHealthMonitor) {
                    clearInterval(this.wgHealthMonitor);
                    this.wgHealthMonitor = null;
                }
                this.wgActive = false;
                try {
                    if (process.platform === 'win32') {
                        execSync(`${wgTools.wireguard} /uninstalltunnelservice wg0`, { timeout: 10000 });
                    }
                    else {
                        execSync('wg-quick down wg0', { timeout: 10000 });
                    }
                }
                catch (_) { /* may already be down */ }
                // Revert to deploy URL in case we were on WG URL
                if (this.wgDeployUrl) {
                    config_1.config.server.url = this.wgDeployUrl;
                    config_1.config.server.wsUrl = this.wgDeployWsUrl;
                    try {
                        const configFile = path_1.default.join(path_1.default.dirname(process.execPath), 'agent-config.json');
                        const existing = fs_1.default.existsSync(configFile) ? JSON.parse(fs_1.default.readFileSync(configFile, 'utf8')) : {};
                        fs_1.default.writeFileSync(configFile, JSON.stringify({ ...existing, serverUrl: this.wgDeployUrl, wsUrl: this.wgDeployWsUrl }, null, 2));
                    }
                    catch (_) { }
                    this.wsClient.reconnect();
                }
                this.wsClient.sendMetrics('wireguard:status', { agentId, status: 'disabled' });
                return;
            }
            // Check if WG tunnel is already running with a fresh handshake (< 3 min)
            let tunnelAlreadyHealthy = false;
            try {
                const wgOutput = execSync(`${wgTools.wg} show wg0`, { timeout: 5000 }).toString();
                const hsMatch = wgOutput.match(/latest handshake: (.+)/);
                if (hsMatch) {
                    const hsText = hsMatch[1].trim();
                    // Parse handshake age — "X seconds ago", "X minutes, Y seconds ago", etc.
                    const secMatch = hsText.match(/(\d+)\s*second/);
                    const minMatch = hsText.match(/(\d+)\s*minute/);
                    const totalSec = (minMatch ? parseInt(minMatch[1]) * 60 : 0) + (secMatch ? parseInt(secMatch[1]) : 0);
                    if (totalSec > 0 && totalSec < 180) {
                        tunnelAlreadyHealthy = true;
                        logger_1.logger.info(`WireGuard: tunnel already healthy (handshake ${totalSec}s ago) — skipping restart`);
                        this.wsClient.sendMetrics('wireguard:status', { agentId, status: 'up', assignedIp: peerConfig.assigned_ip, lastHandshake: hsText });
                    }
                }
            }
            catch (_) { /* wg show failed — tunnel not running */ }
            if (tunnelAlreadyHealthy) {
                // Tunnel is already working — switch URLs if not already on WG
                if (!this.wgActive) {
                    this.wgActive = true;
                    this.startWgHealthMonitor(agentId);
                }
                // Derive WG gateway IP and switch URLs
                const subnet = serverConfig.subnet || '';
                const subnetBase = subnet.split('/')[0] || '';
                let wgGatewayIp = subnetBase ? subnetBase.replace(/\.\d+$/, '.1') : null;
                if (!wgGatewayIp || wgGatewayIp.startsWith('0.')) {
                    const assignedIp = String(peerConfig.assigned_ip || '').split('/')[0];
                    wgGatewayIp = assignedIp ? assignedIp.replace(/\.\d+$/, '.1') : null;
                }
                if (wgGatewayIp) {
                    const currentUrl = config_1.config.server.wsUrl || '';
                    if (!currentUrl.includes(wgGatewayIp)) {
                        const deployWsUrl = this.wgDeployWsUrl || config_1.config.server.wsUrl || 'ws://localhost:3001';
                        const wsPort = (() => { try {
                            return new url_1.URL(deployWsUrl).port || '3001';
                        }
                        catch {
                            return '3001';
                        } })();
                        const deployHttpUrl = this.wgDeployUrl || config_1.config.server.url || 'http://localhost:3000';
                        const httpPort = (() => { try {
                            const u = new url_1.URL(deployHttpUrl);
                            return u.port || (u.protocol === 'https:' ? '443' : '80');
                        }
                        catch {
                            return '3000';
                        } })();
                        const wgHttpUrl = `http://${wgGatewayIp}${httpPort !== '80' ? ':' + httpPort : ''}`;
                        const wgWsUrl = `ws://${wgGatewayIp}:${wsPort}`;
                        logger_1.logger.info(`WireGuard: tunnel healthy — switching to WG URLs: ${wgWsUrl}`);
                        config_1.config.server.url = wgHttpUrl;
                        config_1.config.server.wsUrl = wgWsUrl;
                        try {
                            const configFile = path_1.default.join(path_1.default.dirname(process.execPath), 'agent-config.json');
                            const existing = fs_1.default.existsSync(configFile) ? JSON.parse(fs_1.default.readFileSync(configFile, 'utf8')) : {};
                            fs_1.default.writeFileSync(configFile, JSON.stringify({ ...existing, serverUrl: wgHttpUrl, wsUrl: wgWsUrl }, null, 2));
                        }
                        catch (_) { }
                        setTimeout(() => this.wsClient.reconnect(), 2000);
                    }
                }
                return;
            }
            // Build wg0.conf
            const conf = [
                '[Interface]',
                `PrivateKey = ${peerConfig.private_key}`,
                `Address = ${peerConfig.assigned_ip}/32`,
                serverConfig.dns_servers ? `DNS = ${serverConfig.dns_servers}` : '',
                '',
                '[Peer]',
                `PublicKey = ${serverConfig.server_public_key}`,
                peerConfig.preshared_key ? `PresharedKey = ${peerConfig.preshared_key}` : '',
                `Endpoint = ${serverConfig.server_public_ip}:${serverConfig.listen_port || 51820}`,
                // Split-tunnel: only route the WG subnet through the tunnel.
                // Full-tunnel (0.0.0.0/0) would break the agent's direct connection to
                // the server if the WG server isn't yet reachable.
                `AllowedIPs = ${this.getWgSplitTunnelAllowedIp(serverConfig, peerConfig)}`,
                `PersistentKeepalive = ${serverConfig.keepalive_seconds || 25}`,
            ].filter(l => l !== '').join('\n');
            fs_1.default.mkdirSync(wgDir, { recursive: true });
            fs_1.default.writeFileSync(wgConfPath, conf, { mode: 0o600 });
            logger_1.logger.info(`WireGuard config written to ${wgConfPath}`);
            // Start/restart the tunnel
            try {
                if (process.platform === 'win32') {
                    try {
                        execSync(`${wgTools.wireguard} /uninstalltunnelservice wg0`, { timeout: 5000 });
                    }
                    catch (_) { }
                    execSync(`${wgTools.wireguard} /installtunnelservice "${wgConfPath}"`, { timeout: 15000 });
                }
                else {
                    try {
                        execSync('wg-quick down wg0', { timeout: 5000 });
                    }
                    catch (_) { }
                    execSync('wg-quick up wg0', { timeout: 15000 });
                }
                logger_1.logger.info('WireGuard tunnel started');
                this.wsClient.sendMetrics('wireguard:status', { agentId, status: 'up', assignedIp: peerConfig.assigned_ip });
                // ---- Route agent traffic through the WireGuard tunnel ----
                // We wait for a confirmed WireGuard handshake before switching URLs.
                // This ensures the tunnel is actually working before we route traffic
                // through it. We poll every 10 seconds for up to 120 seconds.
                const subnet = serverConfig.subnet || '';
                const subnetBase = subnet.split('/')[0] || '';
                let wgGatewayIp = subnetBase
                    ? subnetBase.replace(/\.\d+$/, '.1')
                    : null;
                if (!wgGatewayIp || wgGatewayIp.startsWith('0.')) {
                    const assignedIp = String(peerConfig.assigned_ip || '').split('/')[0];
                    wgGatewayIp = assignedIp ? assignedIp.replace(/\.\d+$/, '.1') : null;
                }
                // Poll for WireGuard handshake + report status periodically
                let wgSwitchDone = false;
                let wgPollCount = 0;
                const maxPolls = 12; // 12 × 10s = 120s
                const wgPoller = setInterval(() => {
                    wgPollCount++;
                    try {
                        const wgOutput = execSync(`${wgTools.wg} show`, { timeout: 5000 }).toString();
                        const handshakeMatch = wgOutput.match(/latest handshake: (.+)/);
                        const lastHandshake = handshakeMatch ? handshakeMatch[1].trim() : null;
                        const rxMatch = wgOutput.match(/transfer: ([\d.]+\s*\w+) received,\s*([\d.]+\s*\w+) sent/);
                        const parseBytes = (s) => {
                            const m = s.trim().match(/([\d.]+)\s*(\w+)/);
                            if (!m)
                                return 0;
                            const n = parseFloat(m[1]);
                            const u = m[2].toLowerCase();
                            if (u.startsWith('g'))
                                return Math.round(n * 1e9);
                            if (u.startsWith('m'))
                                return Math.round(n * 1e6);
                            if (u.startsWith('k'))
                                return Math.round(n * 1e3);
                            return Math.round(n);
                        };
                        const rxBytes = rxMatch ? parseBytes(rxMatch[1]) : 0;
                        const txBytes = rxMatch ? parseBytes(rxMatch[2]) : 0;
                        // Report status to server (agent stays on original URL during this phase)
                        this.wsClient.sendMetrics('wireguard:status', {
                            agentId, status: 'up', lastHandshake,
                            assignedIp: peerConfig.assigned_ip, rxBytes, txBytes
                        });
                        const hasHandshake = lastHandshake && !lastHandshake.includes('(none)') && lastHandshake !== 'none';
                        if (hasHandshake && !wgSwitchDone && wgGatewayIp) {
                            // Tunnel is live — switch agent to communicate via WG IP
                            wgSwitchDone = true;
                            clearInterval(wgPoller);
                            logger_1.logger.info(`WireGuard: handshake confirmed (${lastHandshake}) — switching to WG gateway ${wgGatewayIp}`);
                            // Use the deploy URL ports (not the current URL which might already be a WG URL)
                            const deployWsUrl = this.wgDeployWsUrl || config_1.config.server.wsUrl || 'ws://localhost:3001';
                            const wsPort = (() => { try {
                                return new url_1.URL(deployWsUrl).port || '3001';
                            }
                            catch {
                                return '3001';
                            } })();
                            const deployHttpUrl = this.wgDeployUrl || config_1.config.server.url || 'http://localhost:3000';
                            const httpPort = (() => { try {
                                const u = new url_1.URL(deployHttpUrl);
                                return u.port || (u.protocol === 'https:' ? '443' : '80');
                            }
                            catch {
                                return '3000';
                            } })();
                            const wgHttpUrl = `http://${wgGatewayIp}${httpPort !== '80' ? ':' + httpPort : ''}`;
                            const wgWsUrl = `ws://${wgGatewayIp}:${wsPort}`;
                            config_1.config.server.url = wgHttpUrl;
                            config_1.config.server.wsUrl = wgWsUrl;
                            try {
                                const configFile = path_1.default.join(path_1.default.dirname(process.execPath), 'agent-config.json');
                                const existing = fs_1.default.existsSync(configFile) ? JSON.parse(fs_1.default.readFileSync(configFile, 'utf8')) : {};
                                fs_1.default.writeFileSync(configFile, JSON.stringify({ ...existing, serverUrl: wgHttpUrl, wsUrl: wgWsUrl }, null, 2));
                                logger_1.logger.info(`WireGuard: agent-config.json updated with WG gateway URLs`);
                            }
                            catch (cfgErr) {
                                logger_1.logger.warn(`WireGuard: could not persist config file: ${cfgErr.message}`);
                            }
                            // Update browser extension config.json if installed
                            const extConfigPath = path_1.default.join(process.env.PROGRAMDATA || 'C:\\ProgramData', 'AppStats', 'browser-extension', 'unpacked', 'config.json');
                            if (fs_1.default.existsSync(extConfigPath)) {
                                try {
                                    const extCfg = JSON.parse(fs_1.default.readFileSync(extConfigPath, 'utf8'));
                                    extCfg.serverUrl = wgHttpUrl.replace(/\/+$/, '');
                                    fs_1.default.writeFileSync(extConfigPath, JSON.stringify(extCfg, null, 2));
                                    logger_1.logger.info(`WireGuard: browser extension config.json updated → ${wgHttpUrl}`);
                                }
                                catch (extErr) {
                                    logger_1.logger.warn(`WireGuard: could not update extension config: ${extErr.message}`);
                                }
                            }
                            // Reconnect WS through tunnel; fall back to deployUrl if it fails
                            setTimeout(() => {
                                logger_1.logger.info('WireGuard: reconnecting WebSocket through tunnel');
                                this.wsClient.reconnect();
                                setTimeout(() => {
                                    if (!this.wsClient.isConnected()) {
                                        logger_1.logger.warn(`WireGuard: tunnel WS failed after 45s — reverting to deploy URL ${this.wgDeployUrl}`);
                                        config_1.config.server.url = this.wgDeployUrl;
                                        config_1.config.server.wsUrl = this.wgDeployWsUrl;
                                        try {
                                            const configFile = path_1.default.join(path_1.default.dirname(process.execPath), 'agent-config.json');
                                            const existing = fs_1.default.existsSync(configFile) ? JSON.parse(fs_1.default.readFileSync(configFile, 'utf8')) : {};
                                            fs_1.default.writeFileSync(configFile, JSON.stringify({ ...existing, serverUrl: this.wgDeployUrl, wsUrl: this.wgDeployWsUrl }, null, 2));
                                        }
                                        catch (_) { }
                                        this.wsClient.reconnect();
                                        this.wgActive = false;
                                    }
                                    else {
                                        // Connected via WG — start the continuous health monitor
                                        logger_1.logger.info('WireGuard: agent fully connected via tunnel — starting health monitor');
                                        this.wgActive = true;
                                        this.startWgHealthMonitor(agentId);
                                    }
                                }, 45000);
                            }, 2000);
                        }
                        else if (wgPollCount >= maxPolls && !wgSwitchDone) {
                            // Timeout — WG server not reachable; stop polling
                            clearInterval(wgPoller);
                            logger_1.logger.warn('WireGuard: no handshake after 120s — server may not be running. Tunnel installed but agent using original URL.');
                        }
                    }
                    catch (_) {
                        if (wgPollCount >= maxPolls)
                            clearInterval(wgPoller);
                    }
                }, 10000);
            }
            catch (err) {
                logger_1.logger.error(`WireGuard tunnel start failed: ${err.message}`);
                this.wsClient.sendMetrics('wireguard:status', { agentId, status: 'error', error: err.message });
            }
        }
        catch (error) {
            logger_1.logger.error(`WireGuard config handler error: ${error.message}`);
            this.wsClient.sendMetrics('wireguard:status', { agentId, status: 'error', error: error.message });
        }
    }
    async shutdown() {
        if (this.shuttingDown)
            return;
        this.shuttingDown = true;
        logger_1.logger.info('Shutting down...');
        // Stop config UI
        if (this.configUIServer) {
            this.configUIServer.close();
        }
        // Clear all intervals
        this.intervals.forEach(i => clearInterval(i));
        this.probeTimers.forEach(t => clearInterval(t));
        if (this.wgHealthMonitor) {
            clearInterval(this.wgHealthMonitor);
            this.wgHealthMonitor = null;
        }
        if (this.probeFlushTimer) {
            clearInterval(this.probeFlushTimer);
            this.probeFlushTimer = null;
        }
        // Flush remaining data
        const records = this.productivityCollector.flushRecords();
        if (records.length > 0) {
            this.wsClient.sendMetrics('productivity:records', { records });
        }
        const events = this.productivityCollector.flushEvents();
        if (events.length > 0) {
            this.wsClient.sendMetrics('activity:events', { events });
        }
        const screenshots = this.productivityCollector.flushScreenshots();
        if (screenshots.length > 0) {
            this.wsClient.send({
                type: 'productivity:screenshots',
                payload: screenshots,
                agentId: this.wsClient.getAgentId() || undefined,
            });
        }
        // Disconnect
        this.wsClient.disconnect();
        logger_1.logger.info('Agent stopped');
        process.exit(0);
    }
}
// Start the agent
const agent = new AppStatsAgent();
agent.start().catch(err => {
    logger_1.logger.error('Failed to start agent:', err);
    process.exit(1);
});
//# sourceMappingURL=index.js.map