"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.SystemCollector = void 0;
const si = __importStar(require("systeminformation"));
const logger_1 = require("../logger");
let prevNetworkStats = null;
class SystemCollector {
    constructor() {
        this.collecting = false;
    }
    async collect() {
        if (this.collecting) {
            logger_1.logger.debug('System collection already in progress, skipping');
            throw new Error('Collection in progress');
        }
        this.collecting = true;
        try {
            const [cpuLoad, cpuSpeed, cpuTemp, mem, disk, network, gpuData, battery, osInfo] = await Promise.all([
                si.currentLoad().catch(() => null),
                si.cpuCurrentSpeed().catch(() => null),
                si.cpuTemperature().catch(() => null),
                si.mem().catch(() => null),
                si.fsSize().catch(() => []),
                si.networkStats().catch(() => []),
                si.graphics().catch(() => null),
                si.battery().catch(() => null),
                si.osInfo().catch(() => null),
            ]);
            // Merge network stats with interface details (IP, MAC, gateway etc.)
            const [networkStats, networkIfaceDetails] = await Promise.all([
                Promise.resolve(network),
                si.networkInterfaces().catch(() => []),
            ]);
            const defaultGateway = await si.networkGatewayDefault().catch(() => '');
            let dnsServers = [];
            try {
                const dns = require('dns');
                dnsServers = dns.getServers ? dns.getServers() : [];
            }
            catch { }
            const ifaceArr = Array.isArray(networkIfaceDetails) ? networkIfaceDetails : [];
            const statsArr = Array.isArray(networkStats) ? networkStats : [];
            const defaultIface = ifaceArr.find((d) => d.default === true || (d.operstate === 'up' && !d.internal && d.ip4)) || ifaceArr.find((d) => !d.internal && d.ip4) || null;
            const defaultIfaceName = defaultIface?.iface || '';
            const detailByName = new Map();
            ifaceArr.forEach((d) => {
                const key = String(d?.iface || d?.ifaceName || '').trim();
                if (key)
                    detailByName.set(key, d);
            });
            const statsByName = new Map();
            statsArr.forEach((s) => {
                const key = String(s?.iface || s?.ifaceName || '').trim();
                if (key)
                    statsByName.set(key, s);
            });
            const interfaceNames = Array.from(new Set([
                ...Array.from(detailByName.keys()),
                ...Array.from(statsByName.keys()),
            ]));
            const vpnHintRegex = /\b(vpn|wireguard|wg\d*|wintun|openvpn|ovpn|tun\d*|tap\d*|utun\d*|ppp\d*|ipsec|ikev2|nordlynx|protonvpn|tailscale|zerotier|zscaler|zpa|zia|globalprotect|pangp|anyconnect|forticlient|fortinet|pulse\s*secure|checkpoint|hamachi|meshnet|warp)\b/i;
            // Build a default-gateway map per iface (si.networkGatewayDefault for primary, no per-iface API)
            const networkInfo = interfaceNames.map((ifaceName) => {
                const iface = statsByName.get(ifaceName) || {};
                const detail = detailByName.get(ifaceName) || {};
                const descriptor = [ifaceName, detail.ifaceName, detail.type, detail.vendor, detail.model]
                    .filter(Boolean)
                    .join(' ')
                    .toLowerCase();
                const isWg = /\b(wireguard|wg\d*|wintun)\b/i.test(descriptor);
                const isTun = /\b(tun\d*|tap\d*|utun\d*|openvpn|ppp)\b/i.test(descriptor);
                const isVpnName = vpnHintRegex.test(descriptor);
                const isVpn = isWg || isTun || isVpnName;
                const isWifi = /(wi-?fi|wlan\d*|wireless|airport|802\.11)/i.test(descriptor) || String(detail.type || '').toLowerCase() === 'wireless';
                const isLo = /(^|[\s_-])(lo\d*|loopback)([\s_-]|$)/i.test(descriptor) || detail.internal === true;
                let type = 'ethernet';
                let vpnType;
                if (isLo) {
                    type = 'loopback';
                }
                else if (isWifi && !isVpn) {
                    type = 'wifi';
                }
                else if (isVpn) {
                    type = 'vpn';
                    vpnType = isWg ? 'wireguard' : isTun ? 'openvpn' : 'other';
                }
                else if (isWifi) {
                    type = 'wifi';
                }
                return {
                    interface: ifaceName,
                    rxBytes: iface.rx_bytes || 0,
                    txBytes: iface.tx_bytes || 0,
                    rxSec: iface.rx_sec || 0,
                    txSec: iface.tx_sec || 0,
                    speed: iface.speed || detail.speed || 0,
                    operstate: iface.operstate || detail.operstate || 'unknown',
                    type,
                    ...(vpnType ? { vpnType } : {}),
                    ip4: detail.ip4 || undefined,
                    ip6: detail.ip6 || undefined,
                    mac: detail.mac || undefined,
                    gateway: (defaultGateway && ifaceName === defaultIfaceName) ? String(defaultGateway) : undefined,
                    dnsServers: dnsServers.length ? dnsServers : undefined,
                    mtu: detail.mtu || undefined,
                    isInternal: detail.internal === true || isLo,
                };
            }).filter((i) => {
                const hasTraffic = Number(i.rxSec || 0) > 0 || Number(i.txSec || 0) > 0;
                const hasIp = !!(i.ip4 || i.ip6);
                const up = String(i.operstate || '').toLowerCase() === 'up';
                return !!i.interface && (hasTraffic || hasIp || up || i.type === 'vpn' || i.type === 'wifi');
            });
            const routes = this.buildInterfaceRoutes(ifaceArr, String(defaultGateway || ''), defaultIfaceName);
            const metrics = {
                cpu: {
                    load: cpuLoad?.currentLoad || 0,
                    user: cpuLoad?.currentLoadUser || 0,
                    system: cpuLoad?.currentLoadSystem || 0,
                    idle: cpuLoad?.currentLoadIdle || 0,
                    cores: cpuLoad?.cpus?.map((c) => c.load) || [],
                    speed: cpuSpeed?.avg || 0,
                    temperature: cpuTemp?.main !== null ? cpuTemp?.main : undefined,
                },
                memory: {
                    total: mem?.total || 0,
                    used: mem?.used || 0,
                    free: mem?.free || 0,
                    percent: mem ? (mem.used / mem.total) * 100 : 0,
                    swapTotal: mem?.swaptotal || 0,
                    swapUsed: mem?.swapused || 0,
                    swapPercent: mem?.swaptotal ? (mem.swapused / mem.swaptotal) * 100 : 0,
                },
                disk: disk.map((d) => ({
                    fs: d.fs,
                    type: d.type,
                    size: d.size,
                    used: d.used,
                    available: d.available,
                    percent: d.use || 0,
                    mount: d.mount,
                })),
                network: networkInfo,
                networkInterfaces: networkInfo,
                routes,
                uptime: process.uptime(),
                loadAvg: (() => { try {
                    return require('os').loadavg();
                }
                catch {
                    return [0, 0, 0];
                } })(),
            };
            // GPU info
            if (gpuData?.controllers?.length) {
                metrics.gpu = gpuData.controllers.map((g) => ({
                    model: g.model,
                    vendor: g.vendor,
                    vram: g.vram || 0,
                    temperature: g.temperatureGpu || undefined,
                    utilizationGpu: g.utilizationGpu || undefined,
                    utilizationMemory: g.utilizationMemory || undefined,
                }));
            }
            // Battery info
            if (battery?.hasBattery) {
                metrics.battery = {
                    hasBattery: true,
                    isCharging: battery.isCharging,
                    percent: battery.percent,
                    timeRemaining: battery.timeRemaining,
                };
            }
            return metrics;
        }
        finally {
            this.collecting = false;
        }
    }
    async collectQuick() {
        const [cpuLoad, mem] = await Promise.all([
            si.currentLoad().catch(() => null),
            si.mem().catch(() => null),
        ]);
        return {
            cpu: {
                load: cpuLoad?.currentLoad || 0,
                user: cpuLoad?.currentLoadUser || 0,
                system: cpuLoad?.currentLoadSystem || 0,
                idle: cpuLoad?.currentLoadIdle || 0,
                cores: cpuLoad?.cpus?.map((c) => c.load) || [],
                speed: 0,
            },
            memory: {
                total: mem?.total || 0,
                used: mem?.used || 0,
                free: mem?.free || 0,
                percent: mem ? (mem.used / mem.total) * 100 : 0,
                swapTotal: 0,
                swapUsed: 0,
                swapPercent: 0,
            },
        };
    }
    async collectWifi() {
        try {
            const [wifiConns, wifiNets, netIfaces, defaultGw] = await Promise.all([
                si.wifiConnections().catch(() => []),
                si.wifiNetworks().catch(() => []),
                si.networkInterfaces().catch(() => []),
                si.networkGatewayDefault().catch(() => ''),
            ]);
            const conn = wifiConns?.[0];
            if (!conn)
                return null;
            // Find matching network for extra info
            const matchNet = wifiNets?.find((n) => n.bssid === conn.bssid || n.ssid === conn.ssid);
            // Find the wifi interface for IP
            const ifaces = Array.isArray(netIfaces) ? netIfaces : [];
            const connIface = String(conn.iface || '').trim().toLowerCase();
            const vpnHintRegex = /\b(vpn|wireguard|wg\d*|wintun|openvpn|ovpn|tun\d*|tap\d*|utun\d*|ppp\d*|ipsec|ikev2|nordlynx|protonvpn|tailscale|zerotier|zscaler|zpa|zia|globalprotect|pangp|anyconnect|forticlient|fortinet|pulse\s*secure|checkpoint|hamachi|meshnet|warp)\b/i;
            const matchWireless = (i) => {
                const ifaceName = String(i?.iface || i?.ifaceName || '').toLowerCase();
                if (!ifaceName || vpnHintRegex.test(ifaceName))
                    return false;
                if (connIface && (ifaceName === connIface || ifaceName.includes(connIface) || connIface.includes(ifaceName)))
                    return true;
                const type = String(i?.type || '').toLowerCase();
                if (type === 'wireless' || type === 'wifi')
                    return true;
                return /(wi-?fi|wlan\d*|wireless|airport|802\.11)/i.test(ifaceName);
            };
            const wifiIface = ifaces.find((i) => matchWireless(i) && String(i?.ip4 || '').trim())
                || ifaces.find((i) => matchWireless(i));
            // Get DNS servers
            let dnsServers = [];
            try {
                const dns = require('dns');
                dnsServers = dns.getServers ? dns.getServers() : [];
            }
            catch { }
            return {
                ssid: conn.ssid || '',
                bssid: conn.bssid || '',
                rssi: conn.signalLevel ?? matchNet?.signalLevel ?? 0,
                signal_quality: conn.quality ?? matchNet?.quality ?? 0,
                channel: conn.channel ?? matchNet?.channel ?? 0,
                frequency: conn.frequency ?? matchNet?.frequency ?? 0,
                link_speed_mbps: conn.txRate || 0,
                tx_rate_mbps: conn.txRate || 0,
                rx_rate_mbps: conn.rxRate || 0,
                noise: matchNet?.noise || 0,
                security: conn.security || matchNet?.security || '',
                interface_name: conn.iface || wifiIface?.iface || '',
                ip_address: wifiIface?.ip4 || conn.ip4 || '',
                gateway: typeof defaultGw === 'string' ? defaultGw : '',
                dns_servers: dnsServers,
                connection_type: 'wifi',
                phy_link_speed_mbps: wifiIface?.speed || 0,
                errors: 0,
            };
        }
        catch (e) {
            logger_1.logger.debug('WiFi collection failed (may not have wifi adapter)');
            return null;
        }
    }
    async collectIPAddresses() {
        try {
            const ifaces = await si.networkInterfaces().catch(() => []);
            const ifaceArr = Array.isArray(ifaces) ? ifaces : [];
            const ips = [];
            let primary = '';
            for (const iface of ifaceArr) {
                if (iface.ip4 && iface.ip4 !== '127.0.0.1' && !iface.internal) {
                    ips.push(iface.ip4);
                    if (!primary && iface.operstate === 'up')
                        primary = iface.ip4;
                }
                if (iface.ip6 && !iface.ip6.startsWith('fe80') && !iface.internal) {
                    ips.push(iface.ip6);
                }
            }
            if (!primary && ips.length)
                primary = ips[0];
            return { primary, all: ips };
        }
        catch {
            return { primary: '', all: [] };
        }
    }
    async collectPhysicalNetwork() {
        try {
            const ifaces = await si.networkInterfaces().catch(() => []);
            const ifaceArr = Array.isArray(ifaces) ? ifaces : [];
            const wiredIface = ifaceArr.find((i) => i.operstate === 'up' && !i.internal && i.type !== 'wireless' && i.ip4 && i.ip4 !== '127.0.0.1');
            if (!wiredIface)
                return null;
            return {
                type: 'ethernet',
                speed: wiredIface.speed || 0,
                interface_name: wiredIface.iface || '',
                ip: wiredIface.ip4 || '',
            };
        }
        catch {
            return null;
        }
    }
    buildInterfaceRoutes(ifaces, defaultGateway, defaultIfaceName) {
        const routes = [];
        if (defaultGateway) {
            routes.push({
                destination: '0.0.0.0/0',
                gateway: defaultGateway,
                interface: defaultIfaceName || '',
                metric: 1,
                family: 'ipv4',
                isDefault: true,
            });
        }
        for (const iface of ifaces || []) {
            if (!iface || iface.internal)
                continue;
            const name = String(iface.iface || '');
            if (iface.ip4) {
                const cidr = Number(iface.ip4subnet || 24);
                const network = this.networkFromIPv4(String(iface.ip4), cidr);
                routes.push({
                    destination: `${network}/${cidr}`,
                    gateway: 'on-link',
                    interface: name,
                    metric: Number(iface.metric || 0),
                    family: 'ipv4',
                });
            }
            if (iface.ip6) {
                const cidr6 = Number(iface.ip6subnet || 64);
                routes.push({
                    destination: `${iface.ip6}/${cidr6}`,
                    gateway: 'on-link',
                    interface: name,
                    metric: Number(iface.metric || 0),
                    family: 'ipv6',
                });
            }
        }
        const dedup = new Map();
        for (const r of routes) {
            const key = `${r.destination}|${r.gateway}|${r.interface}|${r.family}`;
            if (!dedup.has(key))
                dedup.set(key, r);
        }
        return Array.from(dedup.values()).slice(0, 64);
    }
    networkFromIPv4(ip, cidr) {
        const parts = ip.split('.').map((n) => Number(n));
        if (parts.length !== 4 || parts.some((n) => Number.isNaN(n)))
            return ip;
        const bits = Math.max(0, Math.min(32, cidr));
        const ipNum = ((parts[0] << 24) >>> 0) + ((parts[1] << 16) >>> 0) + ((parts[2] << 8) >>> 0) + (parts[3] >>> 0);
        const mask = bits === 0 ? 0 : ((0xffffffff << (32 - bits)) >>> 0);
        const net = ipNum & mask;
        return [24, 16, 8, 0].map((shift) => (net >>> shift) & 255).join('.');
    }
}
exports.SystemCollector = SystemCollector;
//# sourceMappingURL=system.js.map