"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.SystemEventsCollector = void 0;
const logger_1 = require("../logger");
const child_process_1 = require("child_process");
const fs = __importStar(require("fs"));
const path = __importStar(require("path"));
const os = __importStar(require("os"));
class SystemEventsCollector {
    constructor() {
        // Start collecting from 1 hour ago on first run
        this.lastCollectTime = new Date(Date.now() - 3600000);
    }
    async collect() {
        if (process.platform === 'win32') {
            return this.collectWindows();
        }
        return this.collectLinux();
    }
    collectWindows() {
        return new Promise((resolve) => {
            const since = this.lastCollectTime.toISOString();
            const scriptContent = `
$ErrorActionPreference = 'SilentlyContinue'
$since = [DateTime]::Parse('${since}')
$events = @()

# System log - errors and warnings
try {
  $sysEvents = Get-WinEvent -FilterHashtable @{
    LogName = 'System'
    Level = 1,2,3
    StartTime = $since
  } -MaxEvents 50 -ErrorAction SilentlyContinue
  foreach ($e in $sysEvents) {
    $msgLen = if ($e.Message) { [Math]::Min(500, $e.Message.Length) } else { 0 }
    $msg = if ($e.Message -and $msgLen -gt 0) { ($e.Message -replace '[\\r\\n]+', ' ').Substring(0, $msgLen) } else { '' }
    $lvl = switch($e.Level) { 1 {'critical'} 2 {'error'} 3 {'warning'} default {'info'} }
    $un = if ($e.UserId) { $e.UserId.ToString() } else { '' }
    $events += [PSCustomObject]@{
      timestamp = $e.TimeCreated.ToString('o')
      eventType = 'system'
      source = $e.ProviderName
      eventId = $e.Id
      level = $lvl
      message = $msg
      userName = $un
      computer = $e.MachineName
    }
  }
} catch { }

# Security log - logon events
try {
  $secEvents = Get-WinEvent -FilterHashtable @{
    LogName = 'Security'
    Id = 4624,4634,4625,4647,4672
    StartTime = $since
  } -MaxEvents 50 -ErrorAction SilentlyContinue
  foreach ($e in $secEvents) {
    $type = switch($e.Id) {
      4624 {'logon'} 4634 {'logoff'} 4625 {'logon_failed'}
      4647 {'logoff'} 4672 {'special_logon'}
      default {'security'}
    }
    $msgLen = if ($e.Message) { [Math]::Min(500, $e.Message.Length) } else { 0 }
    $msg = if ($e.Message -and $msgLen -gt 0) { ($e.Message -replace '[\\r\\n]+', ' ').Substring(0, $msgLen) } else { '' }
    $un2 = if ($e.UserId) { $e.UserId.ToString() } else { '' }
    $events += [PSCustomObject]@{
      timestamp = $e.TimeCreated.ToString('o')
      eventType = $type
      source = 'Security'
      eventId = $e.Id
      level = 'info'
      message = $msg
      userName = $un2
      computer = $e.MachineName
    }
  }
} catch { }

# Application log - errors
try {
  $appEvents = Get-WinEvent -FilterHashtable @{
    LogName = 'Application'
    Level = 1,2
    StartTime = $since
  } -MaxEvents 30 -ErrorAction SilentlyContinue
  foreach ($e in $appEvents) {
    $msgLen = if ($e.Message) { [Math]::Min(500, $e.Message.Length) } else { 0 }
    $msg = if ($e.Message -and $msgLen -gt 0) { ($e.Message -replace '[\\r\\n]+', ' ').Substring(0, $msgLen) } else { '' }
    $lvl3 = switch($e.Level) { 1 {'critical'} 2 {'error'} default {'warning'} }
    $un3 = if ($e.UserId) { $e.UserId.ToString() } else { '' }
    $events += [PSCustomObject]@{
      timestamp = $e.TimeCreated.ToString('o')
      eventType = 'application'
      source = $e.ProviderName
      eventId = $e.Id
      level = $lvl3
      message = $msg
      userName = $un3
      computer = $e.MachineName
    }
  }
} catch { }

if ($events.Count -gt 0) {
  $events | ConvertTo-Json -Depth 3 -Compress
} else {
  Write-Output '[]'
}
`;
            // Write script to temp file to avoid escaping issues
            const scriptPath = path.join(os.tmpdir(), 'appstats-sysevents.ps1');
            try {
                fs.writeFileSync(scriptPath, scriptContent, 'utf-8');
            }
            catch (writeErr) {
                logger_1.logger.error(`Failed to write system events script: ${writeErr.message}`);
                resolve([]);
                return;
            }
            (0, child_process_1.execFile)('powershell.exe', ['-NoProfile', '-ExecutionPolicy', 'Bypass', '-File', scriptPath], { timeout: 30000, maxBuffer: 1024 * 1024 }, (err, stdout, stderr) => {
                this.lastCollectTime = new Date();
                if (err) {
                    logger_1.logger.warn(`System events collection error: ${err.message}${stderr ? ' stderr: ' + stderr.substring(0, 200) : ''}`);
                    resolve([]);
                    return;
                }
                try {
                    const raw = stdout.trim();
                    if (!raw || raw === 'null' || raw === '') {
                        resolve([]);
                        return;
                    }
                    let parsed = JSON.parse(raw);
                    if (!Array.isArray(parsed))
                        parsed = [parsed];
                    logger_1.logger.info(`System events collected: ${parsed.length} events`);
                    resolve(parsed);
                }
                catch (parseErr) {
                    logger_1.logger.warn(`System events parse error: ${parseErr.message}, raw length: ${stdout.length}`);
                    resolve([]);
                }
            });
        });
    }
    collectLinux() {
        return new Promise((resolve) => {
            const since = this.lastCollectTime.toISOString();
            (0, child_process_1.execFile)('journalctl', ['--since', since, '-p', 'err', '--output', 'json', '--no-pager', '-n', '50'], { timeout: 15000 }, (err, stdout) => {
                this.lastCollectTime = new Date();
                if (err) {
                    resolve([]);
                    return;
                }
                try {
                    const events = stdout.trim().split('\n')
                        .filter((line) => line.trim())
                        .map((line) => {
                        const j = JSON.parse(line);
                        return {
                            timestamp: new Date(parseInt(j.__REALTIME_TIMESTAMP) / 1000).toISOString(),
                            eventType: 'system',
                            source: j.SYSLOG_IDENTIFIER || j._COMM || '',
                            eventId: parseInt(j.PRIORITY) || 0,
                            level: parseInt(j.PRIORITY) <= 3 ? 'error' : 'warning',
                            message: j.MESSAGE || '',
                            userName: j._UID || '',
                            computer: j._HOSTNAME || '',
                        };
                    });
                    resolve(events);
                }
                catch {
                    resolve([]);
                }
            });
        });
    }
}
exports.SystemEventsCollector = SystemEventsCollector;
//# sourceMappingURL=system-events.js.map