"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductivityCollector = void 0;
const logger_1 = require("../logger");
const child_process_1 = require("child_process");
const fs = __importStar(require("fs"));
const os = __importStar(require("os"));
const path = __importStar(require("path"));
// Productivity scoring rules (can be overridden by server config)
const DEFAULT_PRODUCTIVITY_RULES = {
    // Productive
    'code': 'productive', 'visual studio': 'productive', 'terminal': 'productive',
    'powershell': 'productive', 'cmd': 'productive', 'git': 'productive',
    'node': 'productive', 'python': 'productive', 'docker': 'productive',
    'idea': 'productive', 'webstorm': 'productive', 'pycharm': 'productive',
    'sublime': 'productive', 'notepad++': 'productive', 'vim': 'productive',
    'winword': 'productive', 'excel': 'productive', 'powerpnt': 'productive',
    'figma': 'productive', 'photoshop': 'productive', 'illustrator': 'productive',
    // Neutral
    'teams': 'neutral', 'slack': 'neutral', 'zoom': 'neutral',
    'outlook': 'neutral', 'thunderbird': 'neutral', 'discord': 'neutral',
    'explorer': 'neutral', 'finder': 'neutral', 'chrome': 'neutral',
    'firefox': 'neutral', 'edge': 'neutral', 'msedge': 'neutral',
    // Unproductive
    'spotify': 'unproductive', 'netflix': 'unproductive', 'youtube': 'unproductive',
    'steam': 'unproductive', 'epic': 'unproductive',
};
class ProductivityCollector {
    constructor() {
        this.currentApp = null;
        this.currentTitle = null;
        this.appStartTime = Date.now();
        this.lastCpuSecByPid = new Map();
        this.records = [];
        this.events = [];
        this.lastActivityTime = Date.now();
        this.lastIdleState = false;
        this.idleThreshold = 300000; // 5 minutes
        this.segmentDurationMs = 30000; // Emit records every 30s even without app switches
        this.productivityRules = { ...DEFAULT_PRODUCTIVITY_RULES };
        // Granular tracking config — all on by default
        this.trackingMode = 'full';
        this.trackActiveWindow = true;
        this.trackKeystrokesActivity = true;
        this.trackKeystrokesLog = false; // opt-in only — logs actual keystrokes
        this.trackMouseActivity = true;
        this.trackBrowserUrls = true;
        this.trackBrowserTitles = true;
        this.trackNetworkTraffic = true;
        this.trackScreenshots = false;
        this.screenshotIntervalSeconds = 60;
        this.lastScreenshotAt = 0;
        this.screenshots = [];
        this.privacyAnonUsers = false;
        this.privacyBlurTitles = false;
        this.privacyExcludeUrls = false;
    }
    /** Called when server pushes a config update with tenant tracking settings */
    applyTrackingConfig(cfg) {
        const bool = (v, def) => {
            if (v === undefined || v === null)
                return def;
            if (typeof v === 'boolean')
                return v;
            if (typeof v === 'number')
                return v === 1;
            const s = String(v).trim().toLowerCase();
            if (s === '1' || s === 'true' || s === 'yes' || s === 'on')
                return true;
            if (s === '0' || s === 'false' || s === 'no' || s === 'off')
                return false;
            return def;
        };
        if (cfg.tracking_mode !== undefined) {
            this.trackingMode = cfg.tracking_mode || 'off';
        }
        this.trackActiveWindow = bool(cfg.track_active_window, true);
        this.trackKeystrokesActivity = bool(cfg.track_keystrokes_activity, true);
        this.trackKeystrokesLog = bool(cfg.track_keystrokes_log, false);
        this.trackMouseActivity = bool(cfg.track_mouse_activity, true);
        this.trackBrowserUrls = bool(cfg.track_browser_urls, true);
        this.trackBrowserTitles = bool(cfg.track_browser_titles, true);
        this.trackNetworkTraffic = bool(cfg.track_network_traffic, true);
        this.trackScreenshots = bool(cfg.track_screenshots, false);
        const intervalSeconds = Number(cfg.screenshot_interval_seconds);
        if (Number.isFinite(intervalSeconds) && intervalSeconds > 0) {
            this.screenshotIntervalSeconds = Math.floor(intervalSeconds);
        }
        this.privacyAnonUsers = bool(cfg.privacy_anon_users, false);
        this.privacyBlurTitles = bool(cfg.privacy_blur_titles, false);
        this.privacyExcludeUrls = bool(cfg.privacy_exclude_urls, false);
        logger_1.logger.info(`Tracking config applied: mode=${this.trackingMode}, activeWindow=${this.trackActiveWindow}, ksActivity=${this.trackKeystrokesActivity}, ksLog=${this.trackKeystrokesLog}, screenshots=${this.trackScreenshots}`);
    }
    getTrackingMode() { return this.trackingMode; }
    isTrackNetworkTraffic() { return this.trackNetworkTraffic && this.trackingMode !== 'off'; }
    isTrackBrowserUrls() { return this.trackBrowserUrls && this.trackingMode !== 'off'; }
    isTrackBrowserTitles() { return this.trackBrowserTitles && this.trackingMode !== 'off'; }
    async collectActiveWindow() {
        // Respect tracking mode
        if (this.trackingMode === 'off')
            return;
        try {
            const window = this.trackActiveWindow ? await this.getActiveWindow() : null;
            if (!window) {
                if (this.trackScreenshots) {
                    const nowTs = Date.now();
                    const minIntervalMs = Math.max(30, this.screenshotIntervalSeconds) * 1000;
                    if (nowTs - this.lastScreenshotAt >= minIntervalMs) {
                        this.lastScreenshotAt = nowTs;
                        const shot = await this.captureScreenshot(null);
                        if (shot) {
                            this.screenshots.push(shot);
                            if (this.screenshots.length > 12)
                                this.screenshots = this.screenshots.slice(-12);
                        }
                    }
                }
                return;
            }
            const now = Date.now();
            // Real idle detection using OS-level last input info
            const idleMs = await this.getIdleTime();
            const isIdle = idleMs > this.idleThreshold;
            // Track idle state transitions
            if (isIdle !== this.lastIdleState) {
                this.lastIdleState = isIdle;
                this.recordIdleEvent(isIdle);
                if (!isIdle) {
                    this.lastActivityTime = now;
                }
            }
            if (!isIdle) {
                this.lastActivityTime = now;
            }
            // Detect app switch
            if (window.owner !== this.currentApp || window.title !== this.currentTitle) {
                // Save record for previous app
                if (this.currentApp) {
                    const duration = now - this.appStartTime;
                    if (duration > 1000) { // Only record if > 1 second
                        const record = {
                            timestamp: new Date(this.appStartTime).toISOString(),
                            application: this.currentApp,
                            windowTitle: this.currentTitle || '',
                            category: this.categorize(this.currentApp),
                            productivityScore: this.score(this.currentApp, this.currentTitle || undefined),
                            duration,
                            idle: isIdle,
                        };
                        this.records.push(record);
                    }
                    // Emit activity event
                    this.events.push({
                        timestamp: new Date().toISOString(),
                        type: 'app_switch',
                        details: {
                            from: this.currentApp,
                            fromTitle: this.currentTitle,
                            to: window.owner,
                            toTitle: window.title,
                        },
                    });
                }
                this.currentApp = window.owner;
                this.currentTitle = window.title;
                this.appStartTime = now;
            }
            else if (this.currentApp) {
                // Keep producing records for long-running sessions in the same app/window.
                const duration = now - this.appStartTime;
                if (duration >= this.segmentDurationMs) {
                    const record = {
                        timestamp: new Date(this.appStartTime).toISOString(),
                        application: this.currentApp,
                        windowTitle: this.currentTitle || '',
                        category: this.categorize(this.currentApp),
                        productivityScore: this.score(this.currentApp, this.currentTitle || undefined),
                        duration,
                        idle: isIdle,
                    };
                    this.records.push(record);
                    this.appStartTime = now;
                }
            }
            if (this.trackScreenshots) {
                const nowTs = Date.now();
                const minIntervalMs = Math.max(30, this.screenshotIntervalSeconds) * 1000;
                if (nowTs - this.lastScreenshotAt >= minIntervalMs) {
                    this.lastScreenshotAt = nowTs;
                    const shot = await this.captureScreenshot(window);
                    if (shot) {
                        this.screenshots.push(shot);
                        if (this.screenshots.length > 12)
                            this.screenshots = this.screenshots.slice(-12);
                    }
                }
            }
        }
        catch (e) {
            // Active window detection can fail for various reasons
            logger_1.logger.debug('Active window detection error');
        }
    }
    async getActiveWindow() {
        if (process.platform === 'win32') {
            return this.getActiveWindowWindows();
        }
        else if (process.platform === 'darwin') {
            return this.getActiveWindowMac();
        }
        else {
            return this.getActiveWindowLinux();
        }
    }
    getActiveWindowWindows() {
        return new Promise((resolve) => {
            // Try direct foreground-window API first (best accuracy when running in user session).
            const fgScript = [
                'try {',
                '  Add-Type -TypeDefinition @"',
                'using System;',
                'using System.Runtime.InteropServices;',
                'using System.Text;',
                'public static class FG {',
                '  [DllImport("user32.dll")] public static extern IntPtr GetForegroundWindow();',
                '  [DllImport("user32.dll", CharSet=CharSet.Unicode)] public static extern int GetWindowText(IntPtr hWnd, StringBuilder text, int count);',
                '  [DllImport("user32.dll")] public static extern uint GetWindowThreadProcessId(IntPtr hWnd, out int processId);',
                '}',
                '"@;',
                '  $h = [FG]::GetForegroundWindow();',
                '  if ($h -eq [IntPtr]::Zero) { exit 2 }',
                '  $sb = New-Object System.Text.StringBuilder 1024;',
                '  [void][FG]::GetWindowText($h, $sb, $sb.Capacity);',
                '  $pid = 0;',
                '  [void][FG]::GetWindowThreadProcessId($h, [ref]$pid);',
                '  if ($pid -le 0) { exit 3 }',
                '  $p = Get-Process -Id $pid -ErrorAction SilentlyContinue;',
                '  if (-not $p) { exit 4 }',
                '  "$($sb.ToString())|$($p.ProcessName)|$(if($p.Path){$p.Path}else{""})|$pid"',
                '} catch { exit 1 }',
            ].join(' ');
            (0, child_process_1.exec)(`powershell -NoProfile -Command "${fgScript}"`, { timeout: 5000 }, (fgErr, fgOut) => {
                if (!fgErr && fgOut.trim()) {
                    const parts = fgOut.trim().split('|');
                    if (parts.length >= 2 && (parts[0] || parts[1])) {
                        resolve({
                            title: parts[0] || '',
                            owner: parts[1] || '',
                            path: parts[2] || undefined,
                            pid: parts[3] ? parseInt(parts[3]) : undefined,
                        });
                        return;
                    }
                }
                // Fallback: use tasklist /V and CPU-delta/session/title heuristics from Session 0.
                (0, child_process_1.exec)('tasklist /V /FO CSV /NH', { timeout: 8000 }, (err, stdout) => {
                    if (!err && stdout.trim()) {
                        const result = this.parseTasklistOutput(stdout);
                        if (result) {
                            resolve(result);
                            return;
                        }
                    }
                    // Last fallback: Get-Process window title scan.
                    const script = [
                        '$p = Get-Process |',
                        '  Where-Object { $_.MainWindowTitle -ne \'\' -and $_.SessionId -ne 0 } |',
                        '  Sort-Object CPU -Descending |',
                        '  Select-Object -First 1;',
                        'if ($p) {',
                        '  "$($p.MainWindowTitle)|$($p.Name)|$(if($p.Path){$p.Path}else{""})|$($p.Id)"',
                        '} else {',
                        '  $p2 = Get-Process |',
                        '    Where-Object { $_.MainWindowTitle -ne \'\' } |',
                        '    Sort-Object CPU -Descending |',
                        '    Select-Object -First 1;',
                        '  if ($p2) {',
                        '    "$($p2.MainWindowTitle)|$($p2.Name)|$(if($p2.Path){$p2.Path}else{""})|$($p2.Id)"',
                        '  }',
                        '}',
                    ].join(' ');
                    (0, child_process_1.exec)(`powershell -NoProfile -Command "${script}"`, { timeout: 5000 }, (err2, stdout2) => {
                        if (err2 || !stdout2.trim()) {
                            resolve(null);
                            return;
                        }
                        const parts = stdout2.trim().split('|');
                        if (parts.length >= 2 && (parts[0] || parts[1])) {
                            resolve({
                                title: parts[0] || '',
                                owner: parts[1] || '',
                                path: parts[2] || undefined,
                                pid: parts[3] ? parseInt(parts[3]) : undefined,
                            });
                        }
                        else {
                            resolve(null);
                        }
                    });
                });
            });
        });
    }
    /**
     * Parse `tasklist /V /FO CSV /NH` output.
     * Columns: "Image Name","PID","Session Name","Session#","Mem Usage","Status","User Name","CPU Time","Window Title"
     * We pick the row with the highest CPU time among those with a non-N/A window title
     * in an interactive session (Session# > 0).
     */
    parseTasklistOutput(raw) {
        const lines = raw.split('\n').filter(l => l.trim());
        let bestWithTitle = null;
        let bestWithTitleRank = -Infinity;
        let bestLikelyInteractive = null;
        let bestLikelyInteractiveRank = -Infinity;
        let bestLikelyGlobal = null;
        let bestLikelyGlobalRank = -Infinity;
        let bestAnyInteractive = null;
        let bestAnyInteractiveRank = -Infinity;
        let bestAnyGlobal = null;
        let bestAnyGlobalRank = -Infinity;
        const nextCpuSecByPid = new Map();
        for (const line of lines) {
            // Parse CSV fields (handle quoted values with commas)
            const fields = [];
            let inQuote = false;
            let field = '';
            for (const ch of line) {
                if (ch === '"') {
                    inQuote = !inQuote;
                }
                else if (ch === ',' && !inQuote) {
                    fields.push(field);
                    field = '';
                }
                else {
                    field += ch;
                }
            }
            fields.push(field);
            if (fields.length < 9)
                continue;
            const [imageName, pidStr, , sessionNum, , , , cpuTime, windowTitle] = fields;
            const title = (windowTitle || '').trim();
            const session = parseInt(sessionNum || '0');
            // Parse CPU time "H:MM:SS" to seconds
            const cpuParts = (cpuTime || '0:00:00').split(':').map(Number);
            const cpuSec = (cpuParts[0] || 0) * 3600 + (cpuParts[1] || 0) * 60 + (cpuParts[2] || 0);
            const owner = (imageName || '').replace(/\.exe$/i, '');
            const normalizedOwner = owner.toLowerCase() === 'olk' ? 'outlook' :
                owner.toLowerCase() === 'winword' ? 'word' :
                    owner.toLowerCase() === 'powerpnt' ? 'powerpoint' :
                        owner;
            const pid = parseInt(pidStr || '0') || undefined;
            if (pid)
                nextCpuSecByPid.set(pid, cpuSec);
            const prevCpuSec = pid ? this.lastCpuSecByPid.get(pid) : undefined;
            const cpuDeltaSec = prevCpuSec === undefined ? 0 : Math.max(0, cpuSec - prevCpuSec);
            const ownerLower = owner.toLowerCase();
            const likelySystemProc = ownerLower === 'system idle process' ||
                ownerLower === 'idle' ||
                ownerLower === 'system' ||
                ownerLower === 'registry' ||
                ownerLower === 'memcompression' ||
                ownerLower === 'services' ||
                ownerLower === 'lsass' ||
                ownerLower === 'wininit' ||
                ownerLower === 'winlogon' ||
                ownerLower === 'csrss' ||
                ownerLower === 'smss' ||
                ownerLower === 'dwm' ||
                ownerLower === 'fontdrvhost' ||
                ownerLower === 'ctfmon' ||
                ownerLower === 'sihost' ||
                ownerLower === 'taskhostw' ||
                ownerLower === 'textinputhost' ||
                ownerLower === 'startmenuexperiencehost' ||
                ownerLower === 'shellexperiencehost' ||
                ownerLower === 'applicationframehost' ||
                ownerLower === 'searchhost' ||
                ownerLower === 'searchprotocolhost' ||
                ownerLower === 'searchfilterhost' ||
                ownerLower === 'runtimebroker' ||
                ownerLower === 'audiodg' ||
                ownerLower === 'wudfhost' ||
                ownerLower === 'spoolsv' ||
                ownerLower.startsWith('svchost');
            if (likelySystemProc)
                continue;
            const likelyBackgroundProc = ownerLower.includes('service') ||
                ownerLower.includes('daemon') ||
                ownerLower.includes('updater') ||
                ownerLower.includes('helper') ||
                ownerLower.includes('agent') ||
                ownerLower.includes('vpn') ||
                ownerLower.includes('wireguard') ||
                ownerLower.startsWith('forti') ||
                ownerLower.startsWith('wmi');
            const likelyUserApp = /(chrome|msedge|edge|firefox|brave|opera|vivaldi|explorer|code|devenv|idea|pycharm|webstorm|notepad|winword|excel|powerpnt|outlook|olk|teams|slack|zoom|discord|whatsapp|chatgpt|mstsc|powershell|cmd|terminal)/.test(ownerLower);
            const fallbackRank = (cpuDeltaSec * 10000) +
                (session > 0 ? 5000 : 0) +
                ((title && title !== 'N/A') ? 2000 : 0) +
                (likelyUserApp ? 1000 : 0) -
                (likelyBackgroundProc ? 1000 : 0) +
                ((this.currentApp && normalizedOwner.toLowerCase() === String(this.currentApp).toLowerCase()) ? 25 : 0);
            if (likelyUserApp && fallbackRank > bestLikelyGlobalRank) {
                bestLikelyGlobalRank = fallbackRank;
                bestLikelyGlobal = {
                    title: (!title || title === 'N/A') ? normalizedOwner : title,
                    owner: normalizedOwner,
                    pid,
                };
            }
            if (fallbackRank > bestAnyGlobalRank) {
                bestAnyGlobalRank = fallbackRank;
                bestAnyGlobal = {
                    title: (!title || title === 'N/A') ? normalizedOwner : title,
                    owner: normalizedOwner,
                    pid,
                };
            }
            if (session === 0)
                continue; // Prefer interactive sessions when available.
            if (likelyUserApp && fallbackRank > bestLikelyInteractiveRank) {
                bestLikelyInteractiveRank = fallbackRank;
                bestLikelyInteractive = {
                    title: (!title || title === 'N/A') ? normalizedOwner : title,
                    owner: normalizedOwner,
                    pid,
                };
            }
            if (fallbackRank > bestAnyInteractiveRank) {
                bestAnyInteractiveRank = fallbackRank;
                bestAnyInteractive = {
                    // Some Windows service contexts return N/A for all titles; fall back to process name.
                    title: (!title || title === 'N/A') ? normalizedOwner : title,
                    owner: normalizedOwner,
                    pid,
                };
            }
            if (title && title !== 'N/A' && session > 0 && fallbackRank > bestWithTitleRank) {
                bestWithTitleRank = fallbackRank;
                bestWithTitle = {
                    title,
                    owner: normalizedOwner,
                    pid,
                };
            }
        }
        this.lastCpuSecByPid = nextCpuSecByPid;
        return bestWithTitle || bestLikelyInteractive || bestAnyInteractive || bestLikelyGlobal || bestAnyGlobal;
    }
    getActiveWindowMac() {
        return new Promise((resolve) => {
            const script = `osascript -e 'tell application "System Events" to get {name, unix id} of first process whose frontmost is true' -e 'tell application "System Events" to get name of front window of first process whose frontmost is true'`;
            (0, child_process_1.exec)(script, { timeout: 3000 }, (err, stdout) => {
                if (err) {
                    resolve(null);
                    return;
                }
                const lines = stdout.trim().split('\n');
                resolve({
                    title: lines[1] || '',
                    owner: lines[0]?.split(',')[0]?.trim() || '',
                });
            });
        });
    }
    getActiveWindowLinux() {
        return new Promise((resolve) => {
            (0, child_process_1.exec)('xdotool getactivewindow getwindowname getwindowpid 2>/dev/null', { timeout: 3000 }, (err, stdout) => {
                if (err) {
                    resolve(null);
                    return;
                }
                const lines = stdout.trim().split('\n');
                resolve({
                    title: lines[0] || '',
                    owner: 'unknown',
                    pid: lines[1] ? parseInt(lines[1]) : undefined,
                });
            });
        });
    }
    async captureScreenshot(activeWindow) {
        if (!this.trackScreenshots)
            return null;
        if (process.platform !== 'win32')
            return null;
        return this.captureScreenshotWindows(activeWindow);
    }
    captureScreenshotWindows(activeWindow) {
        return new Promise((resolve) => {
            const tempFile = path.join(os.tmpdir(), `appstats-shot-${Date.now()}-${Math.random().toString(16).slice(2)}.jpg`);
            const safePath = tempFile.replace(/'/g, "''");
            const script = [
                'try {',
                '  Add-Type -AssemblyName System.Windows.Forms;',
                '  Add-Type -AssemblyName System.Drawing;',
                '  $bounds = [System.Windows.Forms.Screen]::PrimaryScreen.Bounds;',
                '  if ($bounds.Width -le 0 -or $bounds.Height -le 0) { exit 2 }',
                '  $bmp = New-Object System.Drawing.Bitmap $bounds.Width, $bounds.Height;',
                '  $gfx = [System.Drawing.Graphics]::FromImage($bmp);',
                '  $gfx.CopyFromScreen($bounds.Location, [System.Drawing.Point]::Empty, $bounds.Size);',
                '  try {',
                "    $codec = [System.Drawing.Imaging.ImageCodecInfo]::GetImageEncoders() | Where-Object { $_.MimeType -eq 'image/jpeg' };",
                '    if ($codec) {',
                '      $ep = New-Object System.Drawing.Imaging.EncoderParameters 1;',
                '      $ep.Param[0] = New-Object System.Drawing.Imaging.EncoderParameter([System.Drawing.Imaging.Encoder]::Quality, 65L);',
                `      $bmp.Save('${safePath}', $codec, $ep);`,
                '      $ep.Dispose();',
                '    } else {',
                `      $bmp.Save('${safePath}', [System.Drawing.Imaging.ImageFormat]::Jpeg);`,
                '    }',
                '  } catch {',
                `    $bmp.Save('${safePath}', [System.Drawing.Imaging.ImageFormat]::Png);`,
                '  }',
                "  Write-Output ($bounds.Width.ToString() + '|' + $bounds.Height.ToString());",
                '  $gfx.Dispose();',
                '  $bmp.Dispose();',
                '} catch { exit 1 }',
            ].join(' ');
            (0, child_process_1.exec)(`powershell -NoProfile -Command "${script}"`, { timeout: 15000, maxBuffer: 4 * 1024 * 1024 }, (err, stdout) => {
                try {
                    if (err || !fs.existsSync(tempFile)) {
                        try {
                            if (fs.existsSync(tempFile))
                                fs.unlinkSync(tempFile);
                        }
                        catch { }
                        resolve(null);
                        return;
                    }
                    const raw = fs.readFileSync(tempFile);
                    try {
                        fs.unlinkSync(tempFile);
                    }
                    catch { }
                    if (!raw.length) {
                        resolve(null);
                        return;
                    }
                    const dims = String(stdout || '').trim().split('|');
                    const width = dims[0] ? parseInt(dims[0], 10) : undefined;
                    const height = dims[1] ? parseInt(dims[1], 10) : undefined;
                    const safeWidth = typeof width === 'number' && !Number.isNaN(width) ? width : undefined;
                    const safeHeight = typeof height === 'number' && !Number.isNaN(height) ? height : undefined;
                    resolve({
                        timestamp: new Date().toISOString(),
                        mimeType: 'image/jpeg',
                        imageBase64: raw.toString('base64'),
                        width: safeWidth,
                        height: safeHeight,
                        application: activeWindow?.owner || '',
                        windowTitle: activeWindow?.title || '',
                        isForeground: true,
                    });
                }
                catch {
                    try {
                        if (fs.existsSync(tempFile))
                            fs.unlinkSync(tempFile);
                    }
                    catch { }
                    resolve(null);
                }
            });
        });
    }
    /**
     * Get user idle time in milliseconds using OS-level APIs.
     * On Windows, uses GetLastInputInfo via PowerShell.
     * This detects mouse/keyboard/touch inactivity at the OS level.
     */
    getIdleTime() {
        if (process.platform === 'win32') {
            return this.getIdleTimeWindows();
        }
        else if (process.platform === 'darwin') {
            return this.getIdleTimeMac();
        }
        else {
            return this.getIdleTimeLinux();
        }
    }
    getIdleTimeWindows() {
        return new Promise((resolve) => {
            // NOTE: GetLastInputInfo() is per-session and returns garbage/boot-time values
            // when called from Session 0 (Windows service). Instead, check whether any
            // interactive user session is Active via qwinsta. If there is an active session
            // we assume the user is at the keyboard (idle = 0 ms) which is the conservative
            // correct behaviour — the session's own idle timer is not queryable cross-session
            // without impersonation. If no session is active, return a large idle value
            // so the interval is classified as inactive.
            const script = [
                'try {',
                '  $sessions = query session 2>$null;',
                '  $active = $sessions | Where-Object { $_ -match "\\bActive\\b" };',
                '  if ($active) { Write-Output 0 } else { Write-Output 300000 }',
                '} catch { Write-Output 0 }',
            ].join(' ');
            (0, child_process_1.exec)(`powershell -NoProfile -Command "${script}"`, { timeout: 5000 }, (err, stdout) => {
                if (err) {
                    resolve(0);
                    return;
                }
                const ms = parseInt(stdout.trim());
                resolve(isNaN(ms) ? 0 : ms);
            });
        });
    }
    getIdleTimeMac() {
        return new Promise((resolve) => {
            (0, child_process_1.exec)('ioreg -c IOHIDSystem | awk \'/HIDIdleTime/ {print $NF/1000000000; exit}\'', { timeout: 3000 }, (err, stdout) => {
                if (err) {
                    resolve(0);
                    return;
                }
                const sec = parseFloat(stdout.trim());
                resolve(isNaN(sec) ? 0 : sec * 1000);
            });
        });
    }
    getIdleTimeLinux() {
        return new Promise((resolve) => {
            (0, child_process_1.exec)('xprintidle 2>/dev/null || echo 0', { timeout: 3000 }, (err, stdout) => {
                if (err) {
                    resolve(0);
                    return;
                }
                const ms = parseInt(stdout.trim());
                resolve(isNaN(ms) ? 0 : ms);
            });
        });
    }
    categorize(appName) {
        const lower = appName.toLowerCase().replace('.exe', '');
        const categories = {
            'development': ['code', 'visual studio', 'terminal', 'powershell', 'cmd', 'git', 'node', 'python', 'docker', 'idea', 'webstorm', 'pycharm', 'sublime', 'notepad++', 'vim'],
            'communication': ['teams', 'slack', 'zoom', 'outlook', 'thunderbird', 'discord', 'skype', 'telegram'],
            'office': ['winword', 'excel', 'powerpnt', 'onenote', 'libreoffice', 'word'],
            'browser': ['chrome', 'firefox', 'edge', 'msedge', 'safari', 'opera', 'brave'],
            'design': ['figma', 'sketch', 'photoshop', 'illustrator', 'gimp', 'inkscape'],
            'media': ['spotify', 'vlc', 'itunes', 'netflix', 'youtube'],
        };
        for (const [cat, apps] of Object.entries(categories)) {
            if (apps.some(a => lower.includes(a)))
                return cat;
        }
        return 'other';
    }
    score(appName, windowTitle) {
        const lower = appName.toLowerCase().replace('.exe', '');
        // First check window title for more specific scoring (browser titles especially)
        if (windowTitle) {
            const titleLower = windowTitle.toLowerCase();
            // Productive title keywords (dev/work tools in browser)
            const productiveTitleKeywords = [
                'github', 'gitlab', 'bitbucket', 'jira', 'confluence', 'asana',
                'trello', 'notion', 'linear', 'azure devops', 'stack overflow',
                'stackoverflow', 'mdn', 'docs.microsoft', 'developer.mozilla',
                'google docs', 'google sheets', 'google slides', 'sharepoint',
                'salesforce', 'hubspot', 'zendesk', 'servicenow', 'grafana',
                'kibana', 'datadog', 'new relic', 'aws console', 'portal.azure',
                'console.cloud.google', 'terraform', 'ansible', 'jenkins',
                'circleci', 'travis', 'vercel', 'netlify', 'heroku',
                'chatgpt', 'copilot', 'claude', 'gemini',
            ];
            if (productiveTitleKeywords.some(k => titleLower.includes(k)))
                return 'productive';
            // Unproductive title keywords
            const unproductiveTitleKeywords = [
                'youtube', 'netflix', 'twitch', 'reddit', 'facebook', 'instagram',
                'tiktok', 'twitter', 'x.com', 'hulu', 'disney+', 'amazon prime',
                'spotify', 'soundcloud', '9gag', 'imgur', 'pinterest',
            ];
            if (unproductiveTitleKeywords.some(k => titleLower.includes(k)))
                return 'unproductive';
        }
        // Fall back to app name matching
        for (const [key, score] of Object.entries(this.productivityRules)) {
            if (lower.includes(key))
                return score;
        }
        return 'neutral';
    }
    updateRules(rules) {
        this.productivityRules = { ...DEFAULT_PRODUCTIVITY_RULES, ...rules };
        logger_1.logger.info('Productivity rules updated');
    }
    flushRecords() {
        const records = [...this.records];
        this.records = [];
        return records;
    }
    flushEvents() {
        const events = [...this.events];
        this.events = [];
        return events;
    }
    flushScreenshots() {
        const shots = [...this.screenshots];
        this.screenshots = [];
        return shots;
    }
    recordIdleEvent(isIdle) {
        this.events.push({
            timestamp: new Date().toISOString(),
            type: isIdle ? 'idle_start' : 'idle_end',
            details: { duration: isIdle ? 0 : Date.now() - this.lastActivityTime, idleThresholdMs: this.idleThreshold },
        });
    }
    getLastInputState() {
        return {
            lastActivityTime: this.lastActivityTime,
            isIdle: this.lastIdleState,
            idleSinceMs: this.lastIdleState ? (Date.now() - this.lastActivityTime) : 0
        };
    }
    recordURLVisit(url, title, browser) {
        this.events.push({
            timestamp: new Date().toISOString(),
            type: 'url_visit',
            details: { url, title, browser },
        });
    }
}
exports.ProductivityCollector = ProductivityCollector;
//# sourceMappingURL=productivity.js.map